package ca.evermann.joerg.blockchainWFMS.PetriNet;

import java.io.FileNotFoundException;
import java.io.PrintWriter;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.UUID;

import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

public class PetriNet implements Serializable {

	private TreeSet<Transition>					transitions;
	private TreeSet<Place>						places;
	private TreeMap<Transition, TreeSet<Place>>	presets;
	private TreeMap<Transition, TreeSet<Place>>	postsets;
	private	Place								source;
	private Place								sink;
	private String								name;
	private List<String>						constraints;
	@SuppressWarnings("rawtypes")
	private TreeMap<String, Class>				variables;
	private TreeMap<String, Serializable>		initialValues;
	
	public PetriNet(String name) {
		this.name = name;
		init();
	}
	
	/* We need this when reading from XML */
	@SuppressWarnings("rawtypes")
	void init() {
		transitions = new TreeSet<Transition>();
		places = new TreeSet<Place>();
		presets = new TreeMap<Transition, TreeSet<Place>>();
		postsets = new TreeMap<Transition, TreeSet<Place>>();
		constraints = new ArrayList<String>();
		variables = new TreeMap<String, Class>();
		initialValues = new TreeMap<String, Serializable>();
		source = null;
		sink = null;
	}
	
	public String getName() {
		return name;
	}
	
	public void setName(String newName) {
		this.name = newName;
	}
	
	public void addTransition(Transition t, Place[] preset, Place[] postset) {
		transitions.add(t);
		places.addAll(Arrays.asList(preset));
		places.addAll(Arrays.asList(postset));
		TreeSet<Place> tmp = presets.get(t);
		if (tmp == null) {
			presets.put(t,  new TreeSet<Place>(Arrays.asList(preset)));
		} else {
			tmp.addAll(new TreeSet<Place>(Arrays.asList(preset)));
		}
		tmp = postsets.get(t);
		if (tmp == null) {
			postsets.put(t, new TreeSet<Place>(Arrays.asList(postset)));			
		} else {
			tmp.addAll(new TreeSet<Place>(Arrays.asList(postset)));
		}
	}
	
	public void addVariable(String name, String type, String initialValue) {
		try {
			variables.put(name, Class.forName(type));
			if (type.equals("java.lang.Integer")) {
				Integer initValue = 0;
				try {
					initValue = Integer.valueOf(initialValue);
				} catch (NumberFormatException ex) { }
				initialValues.put(name, initValue);
			}
			if (type.equals("java.lang.Long")) {
				Long initValue = 0l;
				try {
					initValue = Long.valueOf(initialValue);
				} catch (NumberFormatException ex) { }
				initialValues.put(name, initValue);
			}
			if (type.equals("java.lang.Float")) {
				Float initValue = 0f;
				try {
					initValue = Float.valueOf(initialValue);
				} catch (NumberFormatException ex) { }
				initialValues.put(name, initValue);
			}
			if (type.equals("java.lang.Double")) {
				Double initValue = 0d;
				try {
					initValue = Double.valueOf(initialValue);
				} catch (NumberFormatException ex) { }
				initialValues.put(name, initValue);
			}
			if (type.equals("java.lang.Short")) {
				Short initValue = 0;
				try {
					initValue = Short.valueOf(initialValue);
				} catch (NumberFormatException ex) { }
				initialValues.put(name, initValue);
			}
			if (type.equals("java.lang.Boolean")) {
				Boolean initValue = false;
				try {
					initValue = Boolean.valueOf(initialValue);
				} catch (NumberFormatException ex) { }
				initialValues.put(name, initValue);
			}
			if (type.equals("java.lang.String")) {
				initialValues.put(name, initialValue);
			}
		} catch (ClassNotFoundException e) {
			variables.remove(name);
		}
	}
	
	@SuppressWarnings("rawtypes")
	public TreeMap<String, Class> getVariables() {
		return variables;
	}

	public TreeMap<String, Serializable> getInitialValues() {
		return initialValues;
	}

	public void setSource(Place t) {
		if (places.contains(t)) {
			source = t;
		}
	}
	
	public void setSink(Place t) {
		if (places.contains(t)) {
			sink = t;
		}
	}
	
	public Place getSource() {
		return source;
	}
	
	public Place getSink() {
		return sink;
	}
	
	public TreeSet<Transition> getTransitions() {
		return transitions;
	}
	
	public TreeSet<Place> getPlaces() {
		return places;
		
	}

	public TreeSet<Place> getPreset(Transition t) {
		return presets.get(t);
	}
	
	public TreeSet<Place> getPostset(Transition t) {
		return postsets.get(t);
	}
	
	public List<String> getConstraints() {
		return constraints;
	}

	public void setConstraints(List<String> constraints) {
		this.constraints = constraints;
	}

	public Transition getTransition(String tname) {
		Transition target = null;
		for (Transition t : transitions) {
			if (t.getName().equals(tname)) {
				target = t;
				break;
			}
		}
		return target;
	}
	
	public Place getPlace(UUID placeId) {
		Place target = null;
		for (Place p : places) {
			if (p.getId().equals(placeId)) {
				target = p;
				break;
			}
		}
		return target;
	}
	/* 
	 * This will create an example net
	 */
	@SuppressWarnings("rawtypes")
	public static PetriNet createExample() {
		PetriNet net = new PetriNet("Example");
		
		Place p1 = new Place();
		Place p2 = new Place();
		Place p3 = new Place();
		Place p4 = new Place();
		Place p5 = new Place();
		Place p6 = new Place();
		Place p7 = new Place();
		Place p8 = new Place();
		
		Transition t1 = new Transition("Take order");
		Transition t2 = new Transition("Check account");
		Transition t3 = new Transition("Pack order");
		Transition t4 = new Transition("Credit check");
		Transition t5 = new Transition("Decline order");
		Transition t6 = new Transition("Dispatch");
		Transition t7 = new Transition("Return stock");

		TreeSet<String> a1inputs = new TreeSet<String>(Arrays.asList(new String[] {}));
		TreeSet<String> a1outputs = new TreeSet<String>(Arrays.asList(new String[] {"var1", "var2"}));
		TreeMap<String, Class> a1types = new TreeMap<String, Class>();
		a1types.put("var1", Integer.class);
		a1types.put("var2", String.class);
		t1.setActivity(new ActivitySpec("any", "localhost", 8000, a1inputs, a1outputs, "", ""));
		
		TreeSet<String> a2inputs = new TreeSet<String>(Arrays.asList(new String[] {"var1", "var2"}));
		TreeSet<String> a2outputs = new TreeSet<String>(Arrays.asList(new String[] {"var3"}));
		TreeMap<String, Class> a2types = new TreeMap<String, Class>();
		a2types.put("var1", Integer.class);
		a2types.put("var2", String.class);
		a2types.put("var3", String.class);
		t2.setActivity(new ActivitySpec("any", "localhost", 8000, a2inputs, a2outputs, "", ""));

		TreeSet<String> a3inputs = new TreeSet<String>(Arrays.asList(new String[] {"var3", "var1"}));
		TreeSet<String> a3outputs = new TreeSet<String>(Arrays.asList(new String[] {"var3"}));
		TreeMap<String, Class> a3types = new TreeMap<String, Class>();
		a3types.put("var1", Integer.class);
		a3types.put("var2", String.class);
		a3types.put("var3", String.class);
		t3.setActivity(new ActivitySpec("any", "localhost", 8000, a3inputs, a3outputs, "", ""));

		TreeSet<String> a4inputs = new TreeSet<String>(Arrays.asList(new String[] {"var1", "var2"}));
		TreeSet<String> a4outputs = new TreeSet<String>(Arrays.asList(new String[] {"var4"}));
		TreeMap<String, Class> a4types = new TreeMap<String, Class>();
		a4types.put("var1", Integer.class);
		a4types.put("var2", String.class);
		a4types.put("var4", Float.class);
		t4.setActivity(new ActivitySpec("any", "localhost", 8000, a4inputs, a4outputs, "", ""));

		TreeSet<String> a5inputs = new TreeSet<String>(Arrays.asList(new String[] {"var3", "var4"}));
		TreeSet<String> a5outputs = new TreeSet<String>(Arrays.asList(new String[] {}));
		TreeMap<String, Class> a5types = new TreeMap<String, Class>();
		a5types.put("var3", String.class);
		a5types.put("var4", Float.class);
		t5.setActivity(new ActivitySpec("any", "localhost", 8000, a5inputs, a5outputs, "", ""));

		TreeSet<String> a6inputs = new TreeSet<String>(Arrays.asList(new String[] {"var1", "var2", "var3", "var4"}));
		TreeSet<String> a6outputs = new TreeSet<String>(Arrays.asList(new String[] {"var1"}));
		TreeMap<String, Class> a6types = new TreeMap<String, Class>();
		a6types.put("var1", Integer.class);
		a6types.put("var2", String.class);
		a6types.put("var3", String.class);
		a6types.put("var4", Float.class);
		t6.setActivity(new ActivitySpec("any", "localhost", 8000, a6inputs, a6outputs, "", ""));

		TreeSet<String> a7inputs = new TreeSet<String>(Arrays.asList(new String[] {"var2", "var3", "var4"}));
		TreeSet<String> a7outputs = new TreeSet<String>(Arrays.asList(new String[] {}));
		TreeMap<String, Class> a7types = new TreeMap<String, Class>();
		a7types.put("var2", String.class);
		a7types.put("var3", String.class);
		a7types.put("var4", Float.class);
		t7.setActivity(new ActivitySpec("any", "localhost", 8000, a7inputs, a7outputs, "", ""));

		net.addTransition(t1, new Place[] {p1}, new Place[] {p2, p3});
		net.addTransition(t2, new Place[] {p2}, new Place[] {p4});
		net.addTransition(t3, new Place[] {p3}, new Place[] {p5});
		net.addTransition(t4, new Place[] {p4, p5}, new Place[] {p6});
		net.addTransition(t5, new Place[] {p6}, new Place[] {p7});
		net.addTransition(t6, new Place[] {p6}, new Place[] {p8});
		net.addTransition(t7, new Place[] {p7}, new Place[] {p8});
		
		net.setSource(p1);
		net.setSink(p8);

		net.addVariable("var1", "java.lang.Integer", "0");
		net.addVariable("var2", "java.lang.String", "I am var2");
		net.addVariable("var3", "java.lang.String", "I am var3");
		net.addVariable("var4", "java.lang.Float", "0.0");
		
		return net;
	}

	@Override
	public int hashCode() {
		return Objects.hash(transitions, places, presets, postsets, source, sink, name);
	}
	
	@Override
	public boolean equals(Object obj) {
		if (obj == null) return false;
		if (this == obj) return true;
		if (getClass() != obj.getClass()) return false;
		return this.hashCode() == obj.hashCode();
	}
	
	public String toString() {
		return new String("Petri net "+name+" with "+Integer.toString(transitions.size())+" transitions and "+Integer.toString(places.size())+" places");
	}

	public boolean writeToFile(String fname) {
		try {
			PrintWriter out = new PrintWriter(fname);

			out.println("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
			out.println("<WorkflowSpecification name=\"" + name + "\">");
			out.println("\t<Places>");
			for (Place p : places) {
				p.writeToFile(out);
			}
			out.println("\t</Places>");
			out.println("\t<Transitions>");
			for (Transition t : transitions) {
				t.writeToFile(out);
			}
			out.println("\t</Transitions>");
			out.println("\t<Source id=\"" + source.getId().toString() + "\"/>");
			out.println("\t<Sink id=\"" + sink.getId().toString() + "\"/>");
			for (Transition t : transitions) {
				if (presets.get(t) != null && presets.get(t).size() > 0) {
					out.println("\t<Preset transition=\""+t.getName()+"\">");
					for (Place p : presets.get(t)) {
						p.writeToFile(out);
					}
					out.println("\t</Preset>");
				}
				if (postsets.get(t) != null && postsets.get(t).size() > 0) {
					out.println("\t<Postset transition=\""+t.getName()+"\">");
					for (Place p : postsets.get(t)) {
						p.writeToFile(out);
					}					
					out.println("\t</Postset>");
				}
			}
			if (variables.size() > 0) {
				out.println("\t\t\t<Variables>");
				for (String name : variables.keySet()) {
					out.println("\t\t\t\t<Variable type=\""+variables.get(name).getName()+"\" init=\""+initialValues.get(name).toString()+"\">" + name + "</Variable>");
				}
				out.println("\t\t\t</Variables>");
			}
			if (constraints.size() > 0) {
				out.println("\t\t\t<Constraints>");
				for (String constraint : constraints) {
					out.println("\t\t\t\t<Constraint>" + constraint + "</Constraint>");
				}
				out.println("\t\t\t</Constraints>");
			}
			out.println("</WorkflowSpecification>");
			out.close();
			return true;
		} catch (FileNotFoundException e) {
			System.err.println("Could not open file "+fname+" for writing workflow specification");
			return false;
		}
	}
	
	public boolean readFromFile(String fname) {

        SAXParserFactory    theFactory = SAXParserFactory.newInstance();
        SAXParser           theParser;
        PetriNetXMLHandler	theHandler = new PetriNetXMLHandler(this);
        
        try {
            theFactory.setValidating(true);
            theParser = theFactory.newSAXParser();
            theParser.parse(fname, theHandler);
            return true;
        }
        catch (Exception e) {
        	System.err.println("Error while reading workflow specification from file "+fname);
        	return false;
        }
	}
}
