package ca.evermann.joerg.blockchainWFMS.chain;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import ca.evermann.joerg.blockchainWFMS.CA.PeerCertificate;
import ca.evermann.joerg.blockchainWFMS.main.BlockChainUtils;
import ca.evermann.joerg.blockchainWFMS.p2p.P2PNode;
import ca.evermann.joerg.blockchainWFMS.p2p.messages.BlockRequestMessage;
import ca.evermann.joerg.blockchainWFMS.p2p.messages.BlockSendMessage;
import ca.evermann.joerg.blockchainWFMS.p2p.messages.BlockchainRequestMessage;
import ca.evermann.joerg.blockchainWFMS.p2p.messages.BlockchainSendMessage;

public class BlockService {

	private P2PNode		p2pnode;
	/*
	 * The actual blockchain
	 */
	private Map<String, Block> 	knownBlocks = java.util.Collections.synchronizedMap(new TreeMap<String, Block>());
	private PeerCertificate		latestSendingPeer = null;
	private byte[]				latestHash; // this is the latest hash from the ordering service
	private byte[]				topHash; // this is the topmost hash (backwards from head)
			boolean				isInitializing = true;
	private Block 				head = null; // this is the latest Block we have
	
	public BlockService(P2PNode p2pnode) {
		this.p2pnode = p2pnode;
	}
	
	public void init() {
		// read the local copy
		readBlockchain();
		// get the latest hash from the ordering service
		// and the topmost hash from what we've got
		getLatestHash();
		// request a whole chunk if missing
		requestBlockchain();
		// and verify it
		if (!isReady())
			verifyBlockchain();
	}
	
	private void getLatestHash() {
		latestHash = p2pnode.orderingService.getHead();
		topHash = this.latestHash;
		while (topHash != null && knownBlocks.containsKey(BlockChainUtils.to64(topHash))) {
			topHash = this.knownBlocks.get(BlockChainUtils.to64(topHash)).getPreviousBlockHash();
		}
	}

	private void verifyBlockchain() {
		System.out.println("[BlockService] Starting verification ...");
		boolean missingBlocks = false;
		String checkHash = latestHash==null?null:BlockChainUtils.to64(latestHash);
		if (checkHash != null) {
			while (knownBlocks.containsKey(checkHash)) {
				if (verifyBlock(knownBlocks.get(checkHash))) {
					checkHash = knownBlocks.get(checkHash).getPreviousBlockHash64();
					if (checkHash == null) 
						// arrived at genesis
						break;
				} else {
					knownBlocks.remove(checkHash);
					missingBlocks = true;
					break;
				}
			}
			if (checkHash != null) {
				missingBlocks = true;
				if (latestSendingPeer == null)
					// if we haven't been sent a block from somewhere already, ask everyone
					p2pnode.sendMessage(new BlockRequestMessage(BlockChainUtils.from64(checkHash)));
				else 
					// else, ask only the peer that we got the last block from
					p2pnode.sendMessageTo(latestSendingPeer, new BlockRequestMessage(BlockChainUtils.from64(checkHash)));
			}
		}
		if (!missingBlocks) {
			head = (latestHash==null)?null:knownBlocks.get(BlockChainUtils.to64(latestHash));
			isInitializing = false;
			System.out.println("[BlockService] Blockchain verified ok!");
		}
	}

	public boolean isReady() {
		return !isInitializing;
	}
	
	@SuppressWarnings("unchecked")
	private void readBlockchain() {
		System.out.println("[BlockService] Starting reading from file ...");
		ObjectInputStream o = null;
		try {
			o = new ObjectInputStream(new FileInputStream("Replica"+p2pnode.whoAmI().getRepId()+".Blockchain"));
			knownBlocks = (Map<String, Block>) o.readObject();
			head = (Block) o.readObject();
			o.close();
			System.out.println("[BlockService] Read blockchain from file: " + knownBlocks.size() + " blocks");
		} catch (IOException e) {
			System.err.println("[BlockService] Error while loading blockchain");
		} catch (ClassNotFoundException e) {
			System.err.println("[BlockService] Class not found while loading blockchain");
		}
	}

	public void persistBlockChain() {
		ObjectOutputStream o;
		try {
			o = new ObjectOutputStream(new FileOutputStream("Replica"+p2pnode.whoAmI().getRepId()+".Blockchain"));
			o.writeObject(knownBlocks);
			o.writeObject(head);
			o.close();
		} catch (IOException e) {
			System.err.println("[BlockService] Error while saving blockchain");
		}
	}

	public Block getMainBranchHead() {
		if (!isReady()) return null;
		return head;
	}
	
	public Block getBlock(byte[] hash) {
		if (!isReady()) return null;
		if (hash==null) return null;
		return knownBlocks.get(BlockChainUtils.to64(hash));
	}
	
	private void append(Block block, PeerCertificate fromPeer) {
		// if we already have it, skip it
		if (knownBlocks.containsKey(block.getHash64())) return;
		// accept it
		knownBlocks.put(block.getHash64(), block);
		if (fromPeer != null) {
			// this has been sent to us from someone else
			// remember where it came from
			latestSendingPeer = fromPeer;
		} else {
			// this is a new block from the ordering service
			// this will be the latest block, our new head
			head = block;
			// and we update the latest hash, just in case we receive a new block
			// during initialization
			latestHash = head.getHash();
			// notify the workflow engine if we are in normal operation
			if (isReady())
				try {
					synchronized(p2pnode.newBlockQueue) {
						while (p2pnode.newBlockQueue.size() >= ca.evermann.joerg.blockchainWFMS.main.BlockChainWFMSConfig.blockQueueSize) {
							p2pnode.newBlockQueue.wait();
						}
						p2pnode.newBlockQueue.put(head);
						p2pnode.newBlockQueue.notifyAll();
					}
				} catch (InterruptedException e) {
					System.err.println("[BlockService] Could not pass block with hash: " + head.getHash64() + " to workflow engine!");
				}
		}
		// print status
		System.out.println("[BlockService] Appended block with hash: " + block.getHash64());
	}
	
	public void receiveBlock(Block b, PeerCertificate fromNode) {
		/*
		 * when fromNode == null we got the block as a new block 
		 * from the ordering service, otherwise, it was sent to 
		 * us by a peer
		 */
		if (fromNode != null) {
			// we must still be in the initialization phase
			if (isReady()) {
				System.err.println("[BlockService] Received block from peer while in ready operation. Discarding.");
				return;
			}
		}
		if (verifyBlock(b)) {
			// append block
			append(b, fromNode);
			// and verify the chain if we're still initializing
			if (!isReady()) 
				verifyBlockchain();
		}
		persistBlockChain();
	}

	public void sendBlockTo(PeerCertificate recipient, byte[] blockHash) {
		if (blockHash != null) {
			if (knownBlocks.containsKey(BlockChainUtils.to64(blockHash))) {
				Block b = knownBlocks.get(BlockChainUtils.to64(blockHash));
				p2pnode.sendMessageTo(recipient, new BlockSendMessage(b));
			}
		}
	}
	
	public void requestBlockchain() {
		if (topHash != null) {
			byte[] haveHash = head==null?null:head.getHash();
			if (!Arrays.equals(topHash, haveHash)) {
				for (int escalation = 1; escalation<=4; escalation++) {
					System.out.println("[BlockService] Requesting blockchain from hash: " + 
											(haveHash!=null?BlockChainUtils.to64(haveHash):"null") + 
											" to hash " +
											BlockChainUtils.to64(topHash) + 
											", attempt "+escalation);
					p2pnode.sendMessageToSome(new BlockchainRequestMessage(haveHash, topHash), 1f-(float)Math.pow(0.8f, escalation));
					// wait a while for replies
					try {
						Thread.sleep((long)(1000*Math.pow(2d, escalation)));
					} catch (InterruptedException e) {}
					if (isReady()) 
						break;
				}
			}
		}
	}
	
	public void receiveBlockchain(List<Block> blocks, PeerCertificate fromNode) {
		if (!isReady()) {
			// we must still be in the initialization phase waiting for some blocks
			if (Arrays.equals(blocks.get(blocks.size()-1).getHash(), topHash)) {
				// the received blocks must end in the requested hash
				for (Block b : blocks) {
					if (verifyBlock(b)) {
						append(b, fromNode);
					}
				}
				// verify the chain integrity
				verifyBlockchain();
				// and persist to file
				persistBlockChain();
			}
		} else {
			System.err.println("[BlockService] Received blockchain fragment from peer while in ready operation. Discarding.");
		}
	}

	public void sendBlockchainTo(PeerCertificate recipient, byte[] lowerHash, byte[] upperHash) {
		ArrayList<Block> blockList = new ArrayList<Block>();
		if (isReady()) {
			Block lowerBlock = (lowerHash == null) ? null : knownBlocks.get(BlockChainUtils.to64(lowerHash));
			Block upperBlock = (upperHash == null) ? null : knownBlocks.get(BlockChainUtils.to64(upperHash));
			Block b;
			// We respond only if we have the requested upper block
			if (upperBlock != null)
			{
				b = upperBlock;
				while (b != null && !b.equals(lowerBlock)) {
					blockList.add(0,  b);
					b = getBlock(b.getPreviousBlockHash());
				}
				p2pnode.sendMessageTo(recipient, new BlockchainSendMessage(blockList));
			}
		}
	}
	
	private boolean verifyBlock(Block block) {
		// correct hashes
		if (!Arrays.equals(block.getHash(), block.calculateHash())) {
			System.err.println("  [BlockService] Block hash fail");
			return false;
		}
		// no transactions
		if (block.getTransaction() == null) {
			System.err.println("  [BlockService] Block has no transactions");
			return false;
		}
		if (block.getPreviousBlockHash() != null) {
			Block prevBlock = getBlock(block.getPreviousBlockHash());
			if (prevBlock != null) {
				if (block.getSequence() - prevBlock.getSequence() != 1) {
					System.err.println("  [BlockService] Block has wrong sequence number");
					return false;
				}
			}
		}
		return true;
	}
	
	@Override
	public String toString() {
		String s = new String("Blockchains:\n");
		Block[] blocks = knownBlocks.values().toArray(new Block[] {});
		Arrays.sort(blocks);
		for (Block n : blocks) {
			s = s.concat(n.toString());
			s = s.concat("\n");
		}
		if (head != null) {
			s = s.concat("Main branch head: ").concat(head.getHash64());
		} else {
			s = s.concat("Main branch head: null");
		}
		return s;
	}

}