package ca.evermann.joerg.blockchainWFMS.chain;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectInputStream;
import java.io.ObjectOutput;
import java.io.ObjectOutputStream;

import bftsmart.tom.MessageContext;
import bftsmart.tom.ServiceReplica;
import bftsmart.tom.server.defaultservices.DefaultSingleRecoverable;
import ca.evermann.joerg.blockchainWFMS.main.BlockChainUtils;
import ca.evermann.joerg.blockchainWFMS.p2p.P2PNode;

public class OrderingServiceServer extends DefaultSingleRecoverable implements Runnable {

	int 					replicaId;
	public ServiceReplica 	replica;

    private byte[]  lastHash = null;
    private long	lastSequence=0;
    
    private P2PNode p2pnode;

	public OrderingServiceServer(int id, P2PNode p2pnode) {
		this.replicaId = id;
		this.p2pnode = p2pnode;
	}

	@Override
	public void run() {
		readState();
		replica = new ServiceReplica(replicaId, this, this);
	}

	private synchronized void readState() {
		ObjectInputStream o = null;
		try {
			o = new ObjectInputStream(new FileInputStream("Replica"+replicaId+".OrderingState"));
			byte[] state = (byte[]) o.readObject();
			o.close();
			installSnapshot(state);
		} catch (IOException e) {
			System.err.println("[OrderingServiceServer] IO Exception while loading ordering state");
		} catch (ClassNotFoundException e) {
			System.err.println("[OrderingServiceServer] Class not found while loading ordering state");
		}
	}

	private synchronized void persistState() {
		ObjectOutputStream o;
		try {
			o = new ObjectOutputStream(new FileOutputStream("Replica"+replicaId+".OrderingState"));
			o.writeObject(getSnapshot());
			o.close();
		} catch (IOException e) {
			System.err.println("[OrderingServiceServer] IO Exception while saving ordering state");
		}
	}

	public void stop() {
		replica.kill();
		persistState();
	}
	
	public void restart() {
		readState();
		replica.restart();
	}

	private synchronized void createBlock(Transaction tx) {
		Block b = new Block(lastHash, tx, ++lastSequence);
		lastHash = b.getHash();
		p2pnode.blockService.receiveBlock(b, null);
	}
	
    private byte[] addTx(Transaction transaction) {
    	if (transaction.verifyContent()) {
    		/*
    		 * TODO: When we replay after a state transfer, the workflow engine is
    		 * not typically created and initialized. However, as we replay, we assume
    		 * the transactions are valid.
    		 */
    		if (p2pnode.workflowEngine == null || p2pnode.workflowEngine.validateTransaction(transaction)) {
//				System.out.println((p2pnode.workflowEngine==null?"[OrderingServiceServer] Replayed":"[OrderingServiceServer] Added") + " transaction with hash: " + transaction.getID());
				System.out.println((p2pnode.workflowEngine==null?"[OrderingServiceServer] Replayed":"[OrderingServiceServer] Added") + " transaction " + transaction.payload.toString());
    			/*
    			 * we create a new block and update our result accordingly
    			*/
    			createBlock(transaction);
    		} else {
    			System.err.println("[OrderingServiceServer] Failed to validate transaction " + transaction);
//    			System.exit(-1);
    		}
		}
		/*
		 * TODO: Persisting after every Tx is too much, remove after testing
		*/
		persistState();
		return lastHash;
   	}

	@Override
	public String toString() {
		String s = "Ordering Service\n";
		if (lastHash != null)
			s = s.concat("LastHash: " + BlockChainUtils.to64(lastHash));
		else
			s = s.concat("LastHash: null");
		s = s.concat(" // LastBlock#: " + lastSequence);
		if (replica != null) {
			s = s.concat("\nCurrent View: " + replica.getReplicaContext().getSVController().getCurrentView());
			s = s.concat(" // Current Quorum: " + replica.getReplicaContext().getSVController().getQuorum());
			s = s.concat(" // BFT: " + replica.getReplicaContext().getStaticConfiguration().isBFT());
		} else {
			s = s.concat("Initializing BFT replica");
		}
		return s;
	}
	
	public int getMaxF() {
		if (replica.getReplicaContext().getStaticConfiguration().isBFT())
			return (replica.getReplicaContext().getSVController().getCurrentViewN()-1)/3;
		else
			return (replica.getReplicaContext().getSVController().getCurrentViewN()-1)/2;
	}
	
	// Returns the maximum F if we have one fewer node
	// We need to check this first, before the a leaves
	public int maxFOnLeave() {
		if (replica.getReplicaContext().getStaticConfiguration().isBFT())
			return Math.max(Math.floorDiv(replica.getReplicaContext().getSVController().getCurrentViewN()-2,3),0);
		else
			return Math.max(Math.floorDiv(replica.getReplicaContext().getSVController().getCurrentViewN()-2,2),0);
	}

	public int getCurrentF() {
		return replica.getReplicaContext().getSVController().getCurrentViewF();
	}
	
	public int getCurrentN() {
		return replica.getReplicaContext().getSVController().getCurrentViewN();
	}
	public boolean isBFT() {
		return replica.getReplicaContext().getStaticConfiguration().isBFT();
	}
	
	@Override
	public void installSnapshot(byte[] state) {
		System.out.println("[OrderingServiceServer] Installing ordering service server snapshot.");
		try (ByteArrayInputStream byteIn = new ByteArrayInputStream(state);
				ObjectInput objIn = new ObjectInputStream(byteIn)) {
			lastHash = (byte[]) objIn.readObject();
			lastSequence = objIn.readLong();
			System.out.println("[OrderingServiceServer] New ordering service server state: ");
			System.out.println(this.toString());
		} catch (IOException | ClassNotFoundException e) {
			System.err.println("[OrderingServiceServer] Error while setting ordering service snapshot: "+e.getMessage());
		}
		persistState();
	}

	@Override
	public byte[] getSnapshot() {
		// System.out.println("Fetching ordering service server snapshot.");
		try (ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
				ObjectOutput objOut = new ObjectOutputStream(byteOut)) {
			objOut.writeObject(lastHash);
			objOut.writeLong(lastSequence);
			objOut.flush();
			return byteOut.toByteArray();
		} catch (IOException e) {
			System.err.println("[OrderingServiceServer] Error while getting ordering service snapshot: "+e.getMessage());
		}
		return new byte[0];
	}

	@Override
	public byte[] appExecuteOrdered(byte[] command, MessageContext msgCtx) {
		synchronized(p2pnode.blockService) {
			byte[] reply = null;
			boolean hasReply = false;
			try (ByteArrayInputStream byteIn = new ByteArrayInputStream(command);
					ObjectInput objIn = new ObjectInputStream(byteIn);
					ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
					ObjectOutput objOut = new ObjectOutputStream(byteOut);) {
				OrderingServiceRequestType reqType = (OrderingServiceRequestType)objIn.readObject();
				switch (reqType) {
				case ADD_TX:
					Transaction t = (Transaction)objIn.readObject();
					byte[] result = addTx(t);
					objOut.writeObject(result);
					hasReply = true;
					break;
				case GET_HEAD:
					objOut.writeObject(lastHash);
					hasReply = true;
					break;
				default:
					break;
				}
				if (hasReply) {
					objOut.flush();
					byteOut.flush();
					reply = byteOut.toByteArray();
				} else {
					reply = new byte[0];
				}
	
			} catch (IOException | ClassNotFoundException e) {
				System.err.println("[OrderingServiceServer] Error during appExecuteOrdered" + e.getMessage());
			}
			return reply;
		}
	}

	@Override
	public byte[] appExecuteUnordered(byte[] command, MessageContext msgCtx) {
		byte[] reply = null;
		boolean hasReply = false;
		try (ByteArrayInputStream byteIn = new ByteArrayInputStream(command);
				ObjectInput objIn = new ObjectInputStream(byteIn);
				ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
				ObjectOutput objOut = new ObjectOutputStream(byteOut);) {
			OrderingServiceRequestType reqType = (OrderingServiceRequestType)objIn.readObject();
			switch (reqType) {
			case ADD_TX:
				System.err.println("[OrderingServiceServer] Error during appExecuteUnordered: ADD_TX called");
				break;
			case GET_HEAD:
				System.err.println("[OrderingServiceServer] Error during appExecuteUnordered: GET_HEAD called");
				break;
			default:
				break;
			}
			if (hasReply) {
				objOut.flush();
				byteOut.flush();
				reply = byteOut.toByteArray();
			} else {
				reply = new byte[0];
			}

		} catch (IOException | ClassNotFoundException e) {
			System.err.println("[OrderingServiceServer] Error during appExecuteOrdered" + e.getMessage());
		}
		return reply;
	}

}
