package ca.evermann.joerg.blockchainWFMS.p2p;

import java.io.IOException;
import java.security.PrivateKey;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Random;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.LinkedBlockingQueue;

import bftsmart.reconfiguration.ViewManager;
import bftsmart.reconfiguration.util.Configuration;
import bftsmart.reconfiguration.util.HostsConfig;
import ca.evermann.joerg.blockchainWFMS.CA.PeerCertificate;
import ca.evermann.joerg.blockchainWFMS.chain.Block;
import ca.evermann.joerg.blockchainWFMS.chain.BlockService;
import ca.evermann.joerg.blockchainWFMS.chain.OrderingService;
import ca.evermann.joerg.blockchainWFMS.chain.OrderingServiceServer;
import ca.evermann.joerg.blockchainWFMS.chain.Transaction;
import ca.evermann.joerg.blockchainWFMS.main.BlockChainWFMSConfig;
import ca.evermann.joerg.blockchainWFMS.p2p.messages.BlockSendMessage;
import ca.evermann.joerg.blockchainWFMS.p2p.messages.BlockchainSendMessage;
import ca.evermann.joerg.blockchainWFMS.p2p.messages.Message;
import ca.evermann.joerg.blockchainWFMS.workflow.WorkflowEngine;

public class P2PNode {

	private InboundServer inServer;
	private OutboundServer outServer;
	private InboundMsgHandler msgHandler;
	private Thread inServerThread;
	private Thread outServerThread;
	private Thread msgHandlerThread;
	private Thread workflowEngineThread;

	private ArrayList<PeerCertificate> knownPeers;
			CopyOnWriteArrayList<BlockingQueue<Message>> inboundQueues;
			HashMap<PeerCertificate, BlockingQueue<Message>> outboundQueues;
			HashMap<PeerCertificate, PeerConnection> activeConnections;
	
	private PrivateKey mySigningKey;
	private String signatureAlgorithm;
	private int replicaId;

	public OrderingServiceServer orderingServiceServer;
	private Thread orderingServiceServerThread;
	
	public OrderingService orderingService;
	public BlockService blockService;
	public WorkflowEngine workflowEngine;
	
	public BlockingQueue<Block> newBlockQueue = new LinkedBlockingQueue<Block>();
	
/* 
 * Startup and Shutdown 
 * 
 * */

	public P2PNode(int replicaNum) {
		replicaId = replicaNum;

        HostsConfig hosts = new HostsConfig("", "");
        Configuration conf = new Configuration(replicaId, null);        
		mySigningKey = conf.getPrivateKey();
		signatureAlgorithm = conf.getSignatureAlgorithm();
        knownPeers = new ArrayList<PeerCertificate>();
		for (int i=0; i<hosts.getNum(); i++) {
			knownPeers.add(new PeerCertificate(hosts.getHost(i), 
												hosts.getPort(i)+BlockChainWFMSConfig.portDiff, 
												conf.getPublicKey(i),
												i));
		}
	}
	
	public void init(int replicaNum) {

		orderingServiceServer = new OrderingServiceServer(replicaNum, this);
		orderingServiceServerThread = new Thread(orderingServiceServer, "Ordering Service Server Init Thread");
		orderingServiceServerThread.start();
		
		netUp();

		try {
			System.out.println("[P2PNode] Waiting to join ordering view");
			orderingServiceServerThread.join();
			System.out.println("[P2PNode] Joined ordering view");
			
			orderingService = new OrderingService(replicaNum, this);

			blockService = new BlockService(this);
			blockService.init();
			while (!blockService.isReady()) {
				try {
					Thread.sleep(2000);
					System.out.println("[P2PNode] Waiting for block service to catch up ...");
				} catch (InterruptedException e) { }
			}
			
			// workflowEngine = new ca.evermann.joerg.blockchainWFMS.testing.WorkflowEngineTest(this);
			workflowEngine = new WorkflowEngine(this);

			workflowEngineThread = new Thread(workflowEngine, "WorkflowEngine");
			workflowEngineThread.start();
			
		} catch (InterruptedException e1) {
			System.err.println("[P2PNode] OrderingServiceServerThread join interrupted, stopping this node");
			Shutdown();
			System.exit(-1);
		}
	}

	public void restartBFT() {
		orderingServiceServer.restart();
	}

	public void Shutdown() {
		if (workflowEngine != null) {
			workflowEngine.stop = true;
			try {
				workflowEngineThread.join();
			} catch (InterruptedException e) {}
		}
		if (orderingServiceServer.replica != null) {
	        if (orderingServiceServer.getCurrentF() >= 1) {
				if (orderingServiceServer.maxFOnLeave() < orderingServiceServer.getCurrentF()) {
					System.out.println("[P2PNode] Updating consensus f to " + orderingServiceServer.maxFOnLeave());
			        ViewManager viewManager = new ViewManager("", null);
			        viewManager.setF(orderingServiceServer.maxFOnLeave());
			        viewManager.executeUpdates();		        
			    	viewManager.close();
				}
	        }
	        if (orderingServiceServer.getCurrentN() > 1) {
	        	System.out.println("[P2PNode] Leaving view");
		        ViewManager viewManager = new ViewManager("", null);
				viewManager.removeServer(replicaId);
		        viewManager.executeUpdates();		        
		    	viewManager.close();
	        }
        	orderingServiceServer.stop();
		}
		netDown();
		if (blockService != null) {
			blockService.persistBlockChain();
		}
	}

	public void netUp() {
		System.out.println("[P2PNode] Starting P2P network");

		this.inboundQueues = new CopyOnWriteArrayList<BlockingQueue<Message>>();
		this.outboundQueues = new HashMap<PeerCertificate, BlockingQueue<Message>>();
		this.activeConnections = new HashMap<PeerCertificate, PeerConnection>();
		
		try {
			outServer = new OutboundServer(this);
		} catch (IOException e1) {
			e1.printStackTrace();
		}
		outServerThread = new Thread(outServer, "Outbound Server Thread");
		outServerThread.start();
		
		try {
			inServer = new InboundServer(this);
		} catch (IOException e) {
			e.printStackTrace();
		}
		inServerThread = new Thread(inServer, "Inbound Server Thread");
		inServerThread.start();
		
		msgHandler = new InboundMsgHandler(this);
		msgHandlerThread = new Thread(msgHandler, "Inbound Message Handler Thread");
		msgHandlerThread.start();
		
		/*
		 * Wait for a while for peer connections to be made
		 */
		try {
			Thread.sleep(5000);
		} catch (InterruptedException e) {
			System.err.println("[P2PNode] P2P node woken from sleep");
		}
		System.out.println("[P2PNode] P2P network is up");
	}
	
	public void netDown() {
		inServer.stop = true;
		outServer.stop = true;
		msgHandler.stop = true;

		try {
			inServer.serverSocket.close();
		} catch (IOException e1) {
			System.err.println("Error while closing server socket");
		}
		try {
			inServerThread.join();
		} catch (InterruptedException e) {
			System.err.println("Interrupted while joining inServerThread");
		}
		try {
			outServerThread.join();
		} catch (InterruptedException e) {
			System.err.println("Interrupted while joining outServerThread");
		}
		try {
			msgHandlerThread.join();
		} catch (InterruptedException e) {
			System.err.println("Interrupted while joining msgHandlerThread");
		}
	}
/* 
 * Basic Messaging 
 * 
 * */
	
	protected CopyOnWriteArrayList<BlockingQueue<Message>> getInboundQueues() {
		return this.inboundQueues;
	}
	
	public void sendMessageTo(PeerCertificate recipient, Message msg) {
		msg.setSender(whoAmI());
		msg.signContent(mySigningKey, signatureAlgorithm);
		try {
			BlockingQueue<Message> q = outboundQueues.get(recipient);
			if (q != null)
				q.put(msg);
		} catch (InterruptedException e) {
			System.err.println("Interrupted while message sending");
		}
	}

	public void sendMessageToSome(Message msg, float proportion) {
		msg.setSender(whoAmI());
		msg.signContent(mySigningKey, signatureAlgorithm);

		Random r = new Random();
		boolean sent = false;

		for (BlockingQueue<Message> q : outboundQueues.values()) {
			if (r.nextFloat() < proportion) {
				try {
					q.put(msg);
					sent = true;
				} catch (InterruptedException e) {
					System.err.println("Interrupted while broadcasting");
					continue;
				}
			}
		}
		// make sure it got sent to at least one
		if (!sent && outboundQueues.size() > 0) {
			int index = r.nextInt(outboundQueues.size());
			@SuppressWarnings("unchecked")
			BlockingQueue<Message> q = outboundQueues.values().toArray(new BlockingQueue[0])[index];
			try {
				q.put(msg);
			} catch (InterruptedException e) {
				System.err.println("Interrupted while broadcasting");
			}
		}
	}
	
	public void sendMessage(Message msg) {
		sendMessageToSome(msg, 1.0f);
	}
	

	
/* 
 * Peer Management 
 * 
 * */
	
	public PeerCertificate whoAmI() {
		return knownPeers.get(replicaId);
	}
	
	public ArrayList<PeerCertificate> getPeers() {
		return knownPeers;
	}

	
/* 
 * Message Handlers 
 * 
 * */
	public void sendBlockchainTo(PeerCertificate recipient, byte[] lowerHash, byte[] upperHash) {
		blockService.sendBlockchainTo(recipient, lowerHash, upperHash);
	}

	@SuppressWarnings("unchecked")
	public void receiveBlockChain(BlockchainSendMessage msg) {
		blockService.receiveBlockchain( (ArrayList<Block>)msg.payload, msg.getSender() );
	}
	
	public void receiveBlock(BlockSendMessage msg) {
		blockService.receiveBlock( (Block)msg.payload, msg.getSender() );
	}
	
	public void sendBlockTo(PeerCertificate recipient, byte[] blockHash) {
		blockService.sendBlockTo(recipient,  blockHash);
	}
	
/*
 * Transaction Services
 * 
 * */
	public void signTransaction(Transaction t) {
		t.signContent(mySigningKey, signatureAlgorithm);
	}

	public boolean addTransaction(Transaction t) {
		return orderingService.addTx(t);
	}

	public PrivateKey getSigningKey() {
		return mySigningKey;
	}

	public String getSignatureAlgorithm() {
		return signatureAlgorithm;
	}

}
