package ca.evermann.joerg.blockchainWFMS.p2p;

import java.io.*;
import java.net.*;
import java.util.Objects;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;

import ca.evermann.joerg.blockchainWFMS.CA.CryptoUtils;
import ca.evermann.joerg.blockchainWFMS.CA.PeerCertificate;
import ca.evermann.joerg.blockchainWFMS.p2p.messages.Message;

public abstract class PeerConnection implements Runnable {

	public Boolean stop;
	
	protected Socket socket;
	
	protected ObjectInputStream inStream;
	protected ObjectOutputStream outStream;
	private BlockingQueue<Message> inboundQueue, outboundQueue;
	private PeerCertificate otherCertificate;
	private int myReplicaId, otherReplicaId;
			boolean addOtherToView;
	
	protected NodeServer server;
	
	public PeerConnection(int replicaId, NodeServer server) {
		this.stop = false;
		this.socket = null;
		this.server = server;
		this.myReplicaId = replicaId;
		this.inboundQueue = new LinkedBlockingQueue<Message>();
		this.outboundQueue = new LinkedBlockingQueue<Message>();
	}
	
	public boolean OpenAndVerify(boolean addMeToView) {
		if (socket == null) {
			return false;
		}
		if (!socket.isConnected()) {
			return false;
		}
		if (socket.isClosed()) {
			return false;
		}
		try {
			outStream = new ObjectOutputStream(socket.getOutputStream()); outStream.flush();
			outStream.writeInt(myReplicaId); outStream.flush();
			outStream.writeObject(server.node.whoAmI());
			outStream.writeObject(CryptoUtils.signObject(server.node.whoAmI(), server.node.getSigningKey(), server.node.getSignatureAlgorithm()));
			outStream.writeBoolean(addMeToView);
			outStream.flush();
			inStream = new ObjectInputStream(socket.getInputStream());
			otherReplicaId = inStream.readInt();
			otherCertificate = (PeerCertificate) inStream.readObject();
			byte[] signature = (byte[]) inStream.readObject();
			addOtherToView = inStream.readBoolean();
			
			// check if this comes from the right host
			if (!server.node.getPeers().get(otherReplicaId).getHost().equals(socket.getInetAddress().getHostAddress())) return false;
			// check if all the credentials (especially the pubkey) are as I expect them
			if (!server.node.getPeers().get(otherReplicaId).equals(otherCertificate)) return false;
			// verify the signature
			if (!CryptoUtils.verifyObjectSignature(otherCertificate, signature, server.node.getPeers().get(otherReplicaId).getKey(), server.node.getSignatureAlgorithm())) return false;

			return true;
		} catch (IOException | ClassNotFoundException e) {
			System.err.println("Stream opening failed");
			return false;
		}
	}
	
	public BlockingQueue<Message> getInboundQueue() {
		return inboundQueue;
	}
	
	public BlockingQueue<Message> getOutboundQueue() {
		return outboundQueue;
	}
	
	@Override
	public void run() {
		Thread inHandlerThread;
		Thread outHandlerThread;
		
		InboundQueueHandler inQueueHandler = new InboundQueueHandler(inboundQueue, inStream);
		inHandlerThread = new Thread(inQueueHandler, "Queue handler inbound for " + this.otherCertificate.getHost() + ":" + this.otherCertificate.getPort());
		inHandlerThread.start();
		
		OutboundQueueHandler outQueueHandler = new OutboundQueueHandler(outboundQueue, outStream);
		outHandlerThread = new Thread(outQueueHandler, "Queue handler outbound for " +  this.otherCertificate.getHost() + ":" + this.otherCertificate.getPort());
		outHandlerThread.start();

		while (!stop) {
			try {
				Thread.sleep(500);
			} catch (InterruptedException e) { 
				System.err.println("Interrupted while sleeping");
				continue;
			}
			if (!inHandlerThread.isAlive()) {
				break;
			}
			if (!outHandlerThread.isAlive()) {
				break;
			}
		}
		outQueueHandler.stop=true;
		outHandlerThread.interrupt();
		try {
			outHandlerThread.join();
		} catch (InterruptedException e) {
			System.err.println("Interrupted dying outHandlerThread");
		}
		try {
			inQueueHandler.inStream.close();
		} catch (IOException e1) {
			System.err.println("IO Error when closing inStream");
		}
		inHandlerThread.interrupt();
		try {
			inHandlerThread.join();
		} catch (InterruptedException e) {
			System.err.println("Interrupted dying inHandlerThread");
		}
		shutdown();
	}

	protected void shutdown() {
		synchronized(server.node.inboundQueues) {
			server.node.inboundQueues.remove(this.getInboundQueue());
			server.node.outboundQueues.remove(this.otherCertificate);
			if (this.getOtherCertificate() != null) {
				server.node.activeConnections.remove(this.getOtherCertificate());
			}
		}
		if (outStream != null) {
			try {
				outStream.close();
			} catch (IOException e) {
				System.err.println("IOException during shutdown");
			}
		}
		if (inStream != null) {
			try {
				inStream.close();
			} catch (IOException e) {
				System.err.println("IOException during shutdown");
			}
		}
	}
	
	public PeerCertificate getOtherCertificate() {
		return otherCertificate;
	}
	
	public int getOtherReplicaId() {
		return otherReplicaId;
	}
	
	@Override
	public int hashCode() {
		return Objects.hash(myReplicaId, otherReplicaId, otherCertificate);
	}
	
	@Override
	public boolean equals(Object other) {
		if (other instanceof PeerConnection) {
			return other.hashCode() == this.hashCode();
		} else {
			return false;
		}
	}

}
