package ca.evermann.joerg.blockchainWFMS.workflow;

import java.io.File;
import java.io.PrintWriter;
import java.io.Serializable;
import java.io.StringWriter;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeMap;
import java.util.UUID;
import java.util.Vector;

import javax.tools.DiagnosticCollector;
import javax.tools.JavaCompiler;
import javax.tools.JavaFileObject;
import javax.tools.ToolProvider;
import javax.tools.JavaCompiler.CompilationTask;

import ca.evermann.joerg.blockchainWFMS.PetriNet.PetriNet;
import ca.evermann.joerg.blockchainWFMS.PetriNet.Place;
import ca.evermann.joerg.blockchainWFMS.PetriNet.Transition;
import ca.evermann.joerg.blockchainWFMS.main.BlockChainUtils;

public class PetriNetInstance implements Serializable {

	private TreeMap<Place, Integer>			marking;
	private PetriNet						net;
	private UUID							id;
	private TreeMap<String, Serializable> 	values; 
	
	@SuppressWarnings("unchecked")
	public PetriNetInstance(PetriNet net, UUID caseID) {
		this.id = caseID;
		this.net = net;
		this.marking = new TreeMap<Place, Integer>();
		this.values = (TreeMap<String, Serializable>) BlockChainUtils.deepCopy(net.getInitialValues());
		
		for (Place p : this.net.getPlaces()) {
			marking.put(p,  0);
		}
		marking.put(net.getSource(), 1);
	}
	
	public Set<Transition> getEnabled() {
		Set<Transition> enabledSet = new HashSet<Transition>();
		for (Transition t : net.getTransitions()) {
			boolean e = true;
			Set<Place> preset = net.getPreset(t);
			for (Place p : preset) {
				if (marking.get(p) < 1) {
					e = false;
					break;
				}
			}
			if (e == true) {
				enabledSet.add(t);
			}
		}
		return enabledSet;
	}
	
	public UUID getId() {
		return id;
	}
	
	public PetriNet	getNet() {
		return net;
	}
	
	public TreeMap<String, Serializable> getValues() {
		return values;
	}
	
	public boolean isDeadlocked() {
		return (getEnabled().size()==0 && !isDone());
	}
	
	public void fire(Transition t) {
		// check if this is enabled
		boolean e = true;
		for (Place p : net.getPreset(t)) {
			if (marking.get(p) < 1) {
				e = false;
				break;
			}
		}		
		// update marking
		if (e == true) {
			for (Place p : net.getPreset(t)) {
				marking.put(p, marking.get(p)-1);
			}
			for (Place p : net.getPostset(t)) {
				marking.put(p, marking.get(p)+1);
			}
		}
	}
	
//	public boolean isDone() {
//		boolean d = true;
//		for (Place p : net.getPlaces()) {
//			if (p != net.getSink()) {
//				if (marking.get(p) != 0) {
//					d = false;
//					break;
//				}
//			} else {
//				if (marking.get(p) != 1) {
//					d = false;
//					break;
//				}
//			}
//		}
//		return d;
//	}
	public boolean isDone() {
		// This is optimistic, it assumes a sound net!
		return marking.get(net.getSink()) == 1;
	}
	
	public boolean isInitial() {
		boolean d = true;
		for (Place p : net.getPlaces()) {
			if (p != net.getSource()) {
				if (marking.get(p) != 0) {
					d = false;
					break;
				}
			} else {
				if (marking.get(p) != 1) {
					d = false;
					break;
				}
			}
		}
		return d;
	}
	
	@Override
	public int hashCode() {
		return Objects.hash(net.getName(), id);
	}
	
	@Override
	public boolean equals(Object obj) {
		if (obj == null) return false;
		if (this == obj) return true;
		if (getClass() != obj.getClass()) return false;
		return this.hashCode() == obj.hashCode();
	}
	
	public boolean deepEquals(Object obj) {
		return this.equals(obj) && PetriNetInstance.markingsEqual(this.marking, ((PetriNetInstance)obj).marking);
	}
	
	@Override
	public String toString() {
		String r = "PNI: " + net.getName() + "/" + BlockChainUtils.right(id.toString(), 4) + " (";
		for (Map.Entry<Place, Integer> e : marking.entrySet()) {
			r += BlockChainUtils.right(e.getKey().getId().toString(), 4) + "=" + e.getValue().toString()+ ", ";
		}
		r += ")";
		return r;
	}

	private static boolean markingsEqual(TreeMap<Place, Integer> a, TreeMap<Place, Integer> b) {
		if (a == b) return true;
		if (a.size() != b.size()) return false;
		if (!a.keySet().equals(b.keySet())) return false;
		for (Place p : a.keySet()) {
			if (!a.get(p).equals(b.get(p))) return false;
		}
		return true;
	}
	
	@SuppressWarnings("unchecked")
	public boolean isReachableFrom(PetriNetInstance oldPNI) {
		// must at least refer to the same petri net and the same case
		if (!oldPNI.getNet().getName().equals(getNet().getName()))
			return false;
		if (!oldPNI.getId().equals(getId()))
			return false;
		// return true if they are the same instances (''do nothing'' is a way to //reach// a marking)
		if (oldPNI == this) return true;
		// otherwise, check each transition in turn to see if it leads to this marking
		TreeMap<Place, Integer> originalMarking = (TreeMap<Place, Integer>)oldPNI.marking.clone();
		for (Transition t : oldPNI.getEnabled()) {
			oldPNI.fire(t);
			if (markingsEqual(oldPNI.marking, this.marking)) { 
				oldPNI.marking = (TreeMap<Place, Integer>)originalMarking.clone();
				return true;
			}
			oldPNI.marking = (TreeMap<Place, Integer>)originalMarking.clone();
		}
		return false;
	}

	@SuppressWarnings({ "rawtypes", "unchecked" })
	public boolean checkConstraints() {
		if (!getNet().getConstraints().isEmpty()) {
			JavaCompiler compiler = ToolProvider.getSystemJavaCompiler();
			DiagnosticCollector<JavaFileObject> diagnostics = new DiagnosticCollector<JavaFileObject>();
	
			StringWriter writer = new StringWriter();
			PrintWriter out = new PrintWriter(writer);
			out.println("import java.io.Serializable;");
			for (Serializable s : values.values()) {
				out.println("import "+s.getClass().getName()+";");
			}
			out.println("public class ConstraintEval {");
			out.print("  public static boolean eval(");
			Iterator<String> iter = values.keySet().iterator();
			while(iter.hasNext()) {
				String name = iter.next();
				out.print(values.get(name).getClass().getName() + " " + name);
				if (iter.hasNext()) {
					out.print(", ");
				}
			}
			out.println(") {");
			out.println("    boolean r = true;");
			for (String c : getNet().getConstraints()) {
				out.println("    r = r && ("+c+");");
			}
			out.println("    return r;");
			out.println("  }");
			out.println("}");
			out.close();
			JavaFileObject file = new BlockChainUtils.JavaSourceFromString("ConstraintEval", writer.toString());
	
			Iterable<? extends JavaFileObject> compilationUnits = Arrays.asList(file);
			CompilationTask task = compiler.getTask(null, null, diagnostics, null, null, compilationUnits);		    
			boolean success = task.call();
			if (success) {
				try {
					Vector<Class> inputClasses = new Vector<Class>();
					iter = values.keySet().iterator();
					while(iter.hasNext()) {
						String name = iter.next();
						inputClasses.add(values.get(name).getClass());
					}
		            URLClassLoader classLoader = URLClassLoader.newInstance(new URL[] { new File("").toURI().toURL() });
		            Class c = Class.forName("ConstraintEval", true, classLoader);
		            if (c != null) {
		            	Method m = c.getDeclaredMethod("eval", inputClasses.toArray(new Class[] {}));
		            	if (m != null) {
		            		Boolean r = (Boolean) m.invoke(null,values.values().toArray());
		            		return r;
		            	} else {
		            		System.err.println("Constraint evaluation method not found");
		            		return false;
		            	}
		            } else {
		            	System.err.println("Constraint evaluation class not found");
		            	return false;
		            }
				} catch (ClassNotFoundException e) {
					System.err.println("Class not found: " + e);
				} catch (NoSuchMethodException e) {
					System.err.println("No such method: " + e);
				} catch (IllegalAccessException e) {
					System.err.println("Illegal access: " + e);
				} catch (InvocationTargetException e) {
					System.err.println("Invocation target: " + e);
				} catch (MalformedURLException e) {
					System.err.println("Malformed URL: " + e);
				}
				return false;
			}
			System.err.println("Error compiling constraint");
			return false;
		}
		return true;
	}


}
