package ca.evermann.joerg.blockchainWFMS.workflow;

import java.util.HashMap;
import java.util.UUID;

import ca.evermann.joerg.blockchainWFMS.PetriNet.PetriNet;
import ca.evermann.joerg.blockchainWFMS.chain.Block;
import ca.evermann.joerg.blockchainWFMS.chain.Transaction;
import ca.evermann.joerg.blockchainWFMS.p2p.P2PNode;

public class WorkflowEngine implements Runnable {

	private WorklistUI	worklistUI;
	protected P2PNode		p2pnode;
	
	protected HashMap<UUID, PetriNetInstance> runningInstances;
	protected HashMap<String, PetriNet>		knownPetriNets;
	
	public	boolean stop = false;
	
	public WorkflowEngine() { }
	
	public WorkflowEngine(P2PNode p2pnode) {
		this.p2pnode = p2pnode;
		this.worklistUI = new WorklistUI(p2pnode);
		this.runningInstances = new HashMap<UUID, PetriNetInstance>();
		this.knownPetriNets = new HashMap<String, PetriNet>();
		/*
		 * get initial state
		 */
		readBlockChain();
		/*
		 * Show the user interface
		 */
		this.worklistUI.setEnabled(true);
		this.worklistUI.setVisible(true);
	}

	@Override
	public void run() {
		while (!stop) {
			try {
				synchronized(p2pnode.newBlockQueue) {
					while (p2pnode.newBlockQueue.size() == 0 && !stop) {
						p2pnode.newBlockQueue.wait(5000);
					}
					if (stop) break;
					Block nextBlock = p2pnode.newBlockQueue.poll();
					if (nextBlock != null) {
						receiveBlock(nextBlock);
					}
					p2pnode.newBlockQueue.notifyAll();
				}
			} catch (InterruptedException e) { }
		}
	}
	
	/*
	 * This re-reads the entire chain up to the given head node
	 */
	private void readBlockChainTo(Block head) {
		knownPetriNets.clear();
		runningInstances.clear();
		worklistUI.resetModels();
		/*
		 * Must traverse chain twice: We must have the PetriNet definitions before we
		 * can meaningfully read the instance states, as they refer to the Petri nets
		 */
		Block b = head;
		while (b != null) {
			Transaction t = b.getTransaction();
			if (t instanceof ca.evermann.joerg.blockchainWFMS.workflow.transactions.ModelDefinitionTransaction) {
				PetriNet pn = (PetriNet)t.payload;
				// We only get the most recent
				if (!knownPetriNets.containsKey(pn.getName())) {
					knownPetriNets.put(pn.getName(), pn);
					worklistUI.updatePetriNetList(pn);
				}
			}
			if (b.getPreviousBlockHash() != null) {
				b = p2pnode.blockService.getBlock(b.getPreviousBlockHash());
			} else {
				b = null;
			}
		}
		b = head;
		while (b != null) {
			Transaction t = b.getTransaction();
			if (t instanceof ca.evermann.joerg.blockchainWFMS.workflow.transactions.InstanceStateTransaction) {
				PetriNetInstance pni = (PetriNetInstance)t.payload;
				// We only get the most recent
				if (!runningInstances.containsKey(pni.getId())) {
					runningInstances.put(pni.getId(), pni);
					worklistUI.updateWorklist(pni);
				}
			}
			if (b.getPreviousBlockHash() != null) {
				b = p2pnode.blockService.getBlock(b.getPreviousBlockHash());
			} else {
				b = null;
			}
		}
	}
	
	/*
	 * This re-reads the entire chain up to the current main branch head
	 */
	protected void readBlockChain() {
		readBlockChainTo(p2pnode.blockService.getMainBranchHead());
	}

	/*
	 * This is called by the block service when a block is added to the main branch
	 */
	protected void receiveBlock(Block b) {
		if (b != null) {
			// newly considered confirmed block 
			Transaction t = b.getTransaction();
			if (t instanceof ca.evermann.joerg.blockchainWFMS.workflow.transactions.ModelDefinitionTransaction) {
				PetriNet pn = (PetriNet)t.payload;
				if (!knownPetriNets.containsKey(pn.getName())) {
					knownPetriNets.put(pn.getName(), pn);
					worklistUI.updatePetriNetList(pn);
				}
			}
			if (t instanceof ca.evermann.joerg.blockchainWFMS.workflow.transactions.InstanceStateTransaction) {
				// System.out.println("Workflow engine received transaction " + t.toString());
				PetriNetInstance pni = (PetriNetInstance)t.payload;
				runningInstances.put(pni.getId(), pni);
				// System.out.println("Workflow engine updated instances");
				worklistUI.updateWorklist(pni);
				// System.out.println("Workflow engine updated UI");
			}
			try {
				// TODO: @JE: Why is this here (30 Jan 2020)?? Can this be removed?
				Thread.sleep(20);
			} catch (InterruptedException e) { }
		}
	}
	
	public PetriNetInstance getPetriNetInstance(PetriNet pn, UUID caseId) {
		PetriNetInstance pni = runningInstances.get(caseId);
		if (pni == null) return null;
		if (!pni.getNet().equals(pn)) {
			System.err.println("Wrong PN for caseId "+caseId.toString()+". Expected+"+pn.getName()+", got "+pni.getNet().getName());
			return null;
		}
		return pni;
	}
	
	public PetriNet getPetriNet(String pnName) {
		return knownPetriNets.get(pnName);
	}
	
	public boolean validateTransaction(Transaction t) {
		if (t != null) {
			if (t.payload == null) 
				return false;
			switch (t.getClass().getName()) {
			case "ca.evermann.joerg.blockchainWFMS.workflow.transactions.InstanceStateTransaction":
				PetriNetInstance newPNI = (PetriNetInstance)t.payload;
				// petri net must be known to us
				PetriNet pn = this.knownPetriNets.get(newPNI.getNet().getName());
				if (pn == null) {
					return false;
				}
				// find the petri net instance in the chain
				PetriNetInstance currentPNI = getPetriNetInstance(newPNI.getNet(), newPNI.getId());
				if (currentPNI != null) {
					// if we have one, it must be reachable from there
					return newPNI.checkConstraints() && newPNI.isReachableFrom(currentPNI);					
				} else {
					// if we don't have a current state, this must be the initial state
					if (currentPNI == null && newPNI.isInitial()) {
						return newPNI.checkConstraints();
					} else {
						// otherwise, reject
						return false;
					}
				}
			case "ca.evermann.joerg.blockchainWFMS.workflow.transactions.ModelDefinitionTransaction":
				PetriNet	petriNet = (PetriNet) t.payload;
				if (this.knownPetriNets.containsKey(petriNet.getName())) {
					return false;
				}
				return true;
			default:
				return true;
			}
		}
		return false;
	}
	
	@Override
	public String toString() {
		String s = new String("[WorkflowEngine] Running Instance:\n");
		for (UUID u : runningInstances.keySet()) {
			s = s.concat(" " + runningInstances.get(u).toString() + "\n");
		}
		s = s.concat("Known Petri nets:\n");
		for (String st : knownPetriNets.keySet()) {
			s = s.concat(" " + st + "\n");			
		}
		return s;
	}

}
