package ca.evermann.joerg.blockchainWFMS.workflow;

import java.awt.Container;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Image;
import java.awt.event.ActionEvent;
import java.io.IOException;
import java.io.Serializable;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.Map.Entry;
import java.util.Vector;

import javax.imageio.ImageIO;
import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.filechooser.FileFilter;
import javax.swing.filechooser.FileNameExtensionFilter;
import javax.swing.table.DefaultTableModel;
import ca.evermann.joerg.blockchainWFMS.PetriNet.ActivitySpec;
import ca.evermann.joerg.blockchainWFMS.PetriNet.PetriNet;
import ca.evermann.joerg.blockchainWFMS.PetriNet.Transition;
import ca.evermann.joerg.blockchainWFMS.chain.Transaction;
import ca.evermann.joerg.blockchainWFMS.main.BlockChainUtils;
import ca.evermann.joerg.blockchainWFMS.main.BlockChainWFMSConfig;
import ca.evermann.joerg.blockchainWFMS.main.BlockChainWFMSUI;
import ca.evermann.joerg.blockchainWFMS.p2p.P2PNode;
import ca.evermann.joerg.blockchainWFMS.workflow.transactions.InstanceStateTransaction;
import ca.evermann.joerg.blockchainWFMS.workflow.transactions.ModelDefinitionTransaction;

public class WorklistUI extends JFrame {

	private 	Worklist				worklist;
	private 	DefaultTableModel 		worklistModel;
	private 	JTable 					worklistTable;
	private 	JButton					workItemButton;
	private		DefaultTableModel		petriNetModel;
	private		JTable					petriNetTable;
	private		JButton					pnLoadButton;
	private		JButton					pnSaveButton;
	private		DefaultTableModel		caseModel;
	private		JTable					caseTable;
	private		JButton					caseButton;
	
	private		P2PNode					p2pnode;
	
	protected WorklistUI(P2PNode p2pnode) {
		super();
	
		this.p2pnode = p2pnode;
		worklist = new Worklist();
		
		this.setTitle("WorklistUI for Replica " + p2pnode.whoAmI());
		this.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
		this.setSize(600, 600);
		this.setMinimumSize(new Dimension(600, 600));

		Image icon;
		try {
			URL u = BlockChainWFMSUI.class.getResource("icons8-arbeitsablauf-80.png");
			icon = ImageIO.read(u);
			this.setIconImage(icon);
		} catch (IOException e) {
			// proceed without custom icon
		}

		workItemButton = new JButton("Start & Finish Work Item");
		workItemButton.addActionListener((ActionEvent event) -> {
			int selection = worklistTable.getSelectedRow();
			if (selection != -1) {
				ActivityInstance actInstance = worklist.get((String)worklistTable.getValueAt(selection, 1), (String)worklistTable.getValueAt(selection, 2));
				if (actInstance != null) {
					Map<String, Serializable> outputs = doWorkItem(actInstance);
					if (outputs != null) {
						// create a new instance state, update values and fire marking
						PetriNetInstance newPNI = (PetriNetInstance) BlockChainUtils.deepCopy(actInstance.getPNI());
						actInstance.setPNI(newPNI);
						actInstance.setOutputValues(outputs);
						newPNI.fire(actInstance.getTransition());
						// prepare a blockchain transaction and pass to transaction service
						Transaction t = new InstanceStateTransaction(newPNI);
						t.setOriginator(p2pnode.whoAmI());
						p2pnode.signTransaction(t);
						if (!p2pnode.addTransaction(t)) {
							JOptionPane.showMessageDialog(this, "Transaction could not be added.\nThere may be a pending conflicting transaction.\nYou may retry later.", "Transaction Error (Failure)", JOptionPane.ERROR_MESSAGE);
						}
					}
				}
			}
		});
		
		caseButton = new JButton("Launch New Case");
		caseButton.addActionListener((ActionEvent event) -> {
			int selection = petriNetTable.getSelectedRow();
			if (selection != -1) {
				String pname = (String)petriNetTable.getValueAt(selection, 0);
				if (pname != null) {
					PetriNetInstance newPNI = new PetriNetInstance(p2pnode.workflowEngine.getPetriNet(pname), UUID.randomUUID());

					Transaction t = new InstanceStateTransaction(newPNI);
					t.setOriginator(p2pnode.whoAmI());
					p2pnode.signTransaction(t);
					if (!p2pnode.addTransaction(t)) {
						JOptionPane.showMessageDialog(this, "Transaction could not be added.\nThere may be a pending conflicting transaction.\nYou may retry later.", "Transaction Error (Failure)", JOptionPane.ERROR_MESSAGE);
					}
				}
			}
		});

		pnSaveButton = new JButton("Save Workflow Model");
		pnSaveButton.addActionListener((ActionEvent event) -> {
			int selection = petriNetTable.getSelectedRow();
			if (selection != -1) {
				String pname = (String)petriNetTable.getValueAt(selection, 0);
				if (pname != null) {
					JFileChooser fc = new JFileChooser();
					fc.addChoosableFileFilter(new FileNameExtensionFilter("Blockchain WFMS specification (.bws)", "bws"));
					if (fc.showSaveDialog(this) == JFileChooser.APPROVE_OPTION) {
						String fname;
						try {
							fname = fc.getSelectedFile().getCanonicalPath();
							PetriNet pn = p2pnode.workflowEngine.getPetriNet(pname);
							if (pn != null) {
								pn.writeToFile(fname);
							}
						} catch (IOException e) {
							System.err.println("Get canonical path failed when saving workflow model");
						}
					}
				}
			}
		});
		
		pnLoadButton = new JButton("Load & Define Workflow Model");
		pnLoadButton.addActionListener((ActionEvent event) -> {
			JFileChooser fc = new JFileChooser();
			FileFilter ff = new FileNameExtensionFilter("Blockchain WFMS specification (.bws)", "bws");
			fc.addChoosableFileFilter(ff);
			if (fc.showOpenDialog(this) == JFileChooser.APPROVE_OPTION) {
				String pname = fc.getSelectedFile().getName();
				PetriNet pn = new PetriNet(pname);
				try {
					if (pn.readFromFile(fc.getSelectedFile().getCanonicalPath())) {
						Transaction t = new ModelDefinitionTransaction(pn);
						t.setOriginator(p2pnode.whoAmI());
						p2pnode.signTransaction(t);
						if (!p2pnode.addTransaction(t)) {
							JOptionPane.showMessageDialog(this, "Transaction could not be added.\nThere may be a pending conflicting transaction.\nYou may retry later.", "Transaction Error (Failure)", JOptionPane.ERROR_MESSAGE);
						}
					}
				} catch (IOException e) {
					System.err.println("Get canonical path failed when loading workflow model");
				}
			}
		});

		worklistModel = new DefaultTableModel();
		worklistModel.addColumn("PetriNet");
		worklistModel.addColumn("CaseID");
		worklistModel.addColumn("Activity");
		worklistModel.addColumn("Resource");
		worklistTable = new JTable(worklistModel);
		worklistTable.setDefaultEditor(Object.class, null);
		worklistTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		JScrollPane wlScroll = new JScrollPane(worklistTable);
		wlScroll.setMinimumSize(new Dimension(500, 300));

		caseModel = new DefaultTableModel();
		caseModel.addColumn("PetriNet");
		caseModel.addColumn("CaseID");
		caseModel.addColumn("Deadlocked");
		caseModel.addColumn("Ended");
		caseModel.addColumn("Enabled Transitions");
		caseTable = new JTable(caseModel);
		caseTable.setDefaultEditor(Object.class,  null);
		caseTable.setRowSelectionAllowed(false);
		JScrollPane caseScroll = new JScrollPane(caseTable);
		
		petriNetModel = new DefaultTableModel();
		petriNetModel.addColumn("Name");
		petriNetModel.addColumn("Transitions");
		petriNetModel.addColumn("Places");
		petriNetTable = new JTable(petriNetModel);
		petriNetTable.setDefaultEditor(Object.class, null);
		petriNetTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		JScrollPane pnScroll = new JScrollPane(petriNetTable);
		
	    Container pane = getContentPane();
        GridBagLayout gl = new GridBagLayout();
        GridBagConstraints gc = new GridBagConstraints();
        gc.anchor = GridBagConstraints.WEST;
        gc.fill = GridBagConstraints.BOTH;
        pane.setLayout(gl);
        
        gc.gridx = 0;
        gc.gridy = 0;
        gc.weightx = 1;
        gc.weighty = 0;
        pane.add(pnLoadButton, gc);
        gc.gridx = 1;
        gc.gridy = 0;
        gc.weightx = 1;
        gc.weighty = 0;
        pane.add(pnSaveButton, gc);
        gc.gridx = 0;
        gc.gridy = 1;
        gc.weighty = 2;
        gc.gridwidth = 2;
        pane.add(pnScroll, gc);
        gc.gridx = 0;
        gc.gridy = 2;
        gc.weightx = 1;
        gc.weighty = 0;
        gc.gridwidth = 2;
        pane.add(caseButton, gc);
        gc.gridx = 0;
        gc.gridy = 3;
        gc.weighty = 2;
        gc.gridwidth = 2;
        pane.add(caseScroll, gc);
        gc.gridx = 0;
        gc.gridy = 4;
        gc.weightx = 1;
        gc.weighty = 0;
        gc.gridwidth = 2;
        pane.add(workItemButton, gc);
        gc.gridx = 0;
        gc.gridy = 5;
        gc.weighty = 0;
        gc.gridwidth = 2;
        pane.add(wlScroll, gc);
	}
	
    
    protected void removeCaseFromWorklistModel(String uuid) {
		for (int i = 0; i<worklistModel.getRowCount(); i++) {
			if (worklistModel.getValueAt(i, 1).equals(uuid)) {
				worklistModel.removeRow(i);
				break;
			}
		}
    }
    
	protected void addToWorklistModel(ActivityInstance actInstance) {
		Vector<String> row = new Vector<String>();
		row.add(actInstance.getPNName());
		row.add(actInstance.getCaseId().toString());
		row.add(actInstance.getTransition().getName());
		row.add(actInstance.getTransition().getActivity().getRole());
		worklistModel.addRow(row);
	}
	
	protected void removeFromWorkflowModel(PetriNet pn) {
		String pnName = pn.getName();
		for (int i = 0; i<petriNetModel.getRowCount(); i++) {
			if (petriNetModel.getValueAt(i, 0).equals(pnName)) {
				petriNetModel.removeRow(i);
				break;
			}
		}
	}
	
	protected void addToWorkflowModel(PetriNet pn) {
		Vector<String> row = new Vector<String>();
		row.add(pn.getName());
		row.add(Integer.toString(pn.getTransitions().size()));
		row.add(Integer.toString(pn.getPlaces().size()));
		petriNetModel.addRow(row);
	}
	
	protected void addToOrUpdateCaseModel(PetriNetInstance pni) {
		String caseId = pni.getId().toString();
		for (int i = 0; i<caseModel.getRowCount(); i++) {
			if (caseModel.getValueAt(i, 1).equals(caseId)) {
				caseModel.removeRow(i);
				break;
			}
		}
		Vector<String> row = new Vector<String>();
		row.add(pni.getNet().getName());
		row.add(caseId);
		row.add(pni.isDeadlocked() ? "Y" : "N");
		row.add(pni.isDone() ? "Y" : "N");
		row.add(Integer.toString(pni.getEnabled().size()));
		caseModel.addRow(row);
	}
	
	protected void resetModels() {
		worklistModel.setRowCount(0);
		petriNetModel.setRowCount(0);
		caseModel.setRowCount(0);
	}

	void updateWorklist(PetriNetInstance pni) {
		worklist.removeCase(pni.getId().toString());
		removeCaseFromWorklistModel(pni.getId().toString());
		/*
		 * We will for demonstration purposes not remove
		 * done or deadlocked cases
		 * TODO: Change this for production
		 * 
		if (pni.isDone()) {
			runningInstances.remove(pni.getId());
			return;
		}
		if (pni.isDeadlocked()) {
			runningInstances.remove(pni.getId());
			return;
		}
		*/ ;
		for (Transition transition : pni.getEnabled()) {
			ActivitySpec activity = transition.getActivity();
			if (activity.getHost().equals(this.p2pnode.whoAmI().getHost()) && activity.getPort()==this.p2pnode.whoAmI().getPort()) {
				ActivityInstance actInstance = new ActivityInstance(pni, transition);
				if (activity.getExecClass().length()==0 || activity.getExecMethod().length()==0) {
					worklist.put(actInstance);
					addToWorklistModel(actInstance);
				} else {
					Map<String, Serializable> outputs = actInstance.callExecute();
					if (outputs != null) {
						// create a new instance state, update values and fire marking
						PetriNetInstance newPNI = (PetriNetInstance) BlockChainUtils.deepCopy(actInstance.getPNI());
						actInstance.setPNI(newPNI);
						actInstance.setOutputValues(outputs);
						newPNI.fire(actInstance.getTransition());
						// prepare a blockchain transaction and pass to transaction service
						Transaction t = new InstanceStateTransaction(newPNI);
						t.setOriginator(p2pnode.whoAmI());
						p2pnode.signTransaction(t);
						if (!p2pnode.addTransaction(t)) {
							System.err.println("Automatic transition could not be validated.\nThere may be a pending conflicting transaction.\nYou may retry later.");
						}
					}
				}
			}
		}
		addToOrUpdateCaseModel(pni);
	}

	void updatePetriNetList(PetriNet pn) {
		removeFromWorkflowModel(pn);
		addToWorkflowModel(pn);
	}
	
	@SuppressWarnings("rawtypes")
	private HashMap<String, Serializable> doWorkItem(ActivityInstance activityInstance) {
		HashMap<String, Serializable> outputs = new HashMap<String, Serializable>();

		JPanel panel = new JPanel();
		GridBagLayout gl = new GridBagLayout();
		GridBagConstraints gc = new GridBagConstraints();
		gc.anchor = GridBagConstraints.WEST;
		gc.fill = GridBagConstraints.BOTH;
		panel.setLayout(gl);
		
		int y = 0;
		for (Entry<String, Serializable> e : activityInstance.getInputValues().entrySet()) {
			if (!activityInstance.getOutputVariables().contains(e.getKey())) {
				gc.gridx=0;
				gc.gridy=y;
				panel.add(new JLabel(e.getKey()+"("+activityInstance.getPNI().getNet().getVariables().get(e.getKey())+")"), gc);
				gc.gridx=1;
				gc.gridy=y;
				JTextField tf;
				if (e.getValue() != null) {
					tf = new JTextField(e.getValue().toString(), BlockChainWFMSConfig.textFieldWidth);
				} else {
					System.err.println("Warning: Empty input "+e.getKey()+" in activity instance for "+activityInstance.getTransition().getName()+" in net instance ID "+activityInstance.getCaseId());
					tf = new JTextField(BlockChainWFMSConfig.textFieldWidth);				
				}
				tf.setEditable(false);
				panel.add(tf, gc);
				y++;
			}
		}
		HashMap<String, JTextField> fields = new HashMap<String, JTextField>();
		for (String e : activityInstance.getOutputVariables()) {
			gc.gridx=0;
			gc.gridy=y;
			panel.add(new JLabel(e+"("+activityInstance.getPNI().getNet().getVariables().get(e)+")"), gc);
			gc.gridx=1;
			gc.gridy=y;
			JTextField field;
			field = new JTextField(BlockChainWFMSConfig.textFieldWidth);
			field.setEditable(true);
			if (activityInstance.getInputValues().keySet().contains(e)) {
				field.setText(activityInstance.getInputValues().get(e).toString());
			}
			panel.add(field, gc);
			fields.put(e, field);
			y++;
		}

		int result = JOptionPane.showConfirmDialog(this, panel, "doWorkItem: " + activityInstance.getTransition().getName() + " for case " + activityInstance.getCaseId() + " for Workflow " + activityInstance.getPNName(), JOptionPane.OK_CANCEL_OPTION);
		if (result == JOptionPane.OK_OPTION) {
			for (Entry<String, JTextField> e : fields.entrySet()) {
				String varName = e.getKey();
				Serializable newValue = null;
				Class type = activityInstance.getPNI().getNet().getVariables().get(varName);
				if (type == Integer.class) {
					try {
						newValue = Integer.valueOf(fields.get(varName).getText());
					} catch (NumberFormatException ex) {
						newValue = new Integer(0);
					}
				}
				if (type == String.class) {
					newValue = fields.get(varName).getText();
				}
				if (type == Long.class) {
					try {
						newValue = Long.valueOf(fields.get(varName).getText());
					} catch (NumberFormatException ex) {
						newValue = new Long(0l);
					}
				}
				if (type == Float.class) {
					try {
						newValue = Float.valueOf(fields.get(varName).getText());
					} catch (NumberFormatException ex) {
						newValue = new Float(0f);
					}
				}
				if (type == Double.class) {
					try {
						newValue = Double.valueOf(fields.get(varName).getText());
					} catch (NumberFormatException ex) {
						newValue = new Double(0d);
					}
				}
				if (type == Short.class) {
					try {
						newValue = Short.valueOf(fields.get(varName).getText());
					} catch (NumberFormatException ex) {
						newValue = new Short((short)0);
					}
				}
				if (type == Boolean.class) {
					newValue = Boolean.valueOf(fields.get(varName).getText());
				}
				if (newValue == null) {
					System.err.println("Unrecognized type or invalid conversion in doWorkItem");
				}
				outputs.put(varName, newValue);
			}
		}
		if (result == JOptionPane.OK_OPTION)
			return outputs;
		else 
			return null;
	}
		
	@Override
	public String toString() {
		return worklist.toString();
	}

}
