package ca.evermann.joerg.blockchainWFMS.chain;

import java.io.Serializable;
import java.util.Base64;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Queue;

import ca.evermann.joerg.blockchainWFMS.CA.CryptoUtils;

public class Block implements Serializable {

    private byte[] hash;
    private byte[] previousBlockHash;
    private List<Transaction> transactions;
    private byte[] merkleRoot;
    private long timestamp;
    private int miningNonce;

    public Block(byte[] previousBlockHash, List<Transaction> transactions, int miningNonce) {
        this.previousBlockHash = previousBlockHash;
        this.transactions = transactions;
        this.miningNonce = miningNonce;
        this.timestamp = System.currentTimeMillis();
        this.merkleRoot = calculateMerkleRoot();
        this.hash = calculateHash();
    }

    public byte[] getHash() {
        return hash;
    }

    public void setHash(byte[] hash) {
        this.hash = hash;
    }

    public byte[] getPreviousBlockHash() {
        return previousBlockHash;
    }

    public void setPreviousBlockHash(byte[] previousBlockHash) {
        this.previousBlockHash = previousBlockHash;
    }

    public List<Transaction> getTransactions() {
        return transactions;
    }

    public void setTransactions(List<Transaction> transactions) {
        this.transactions = transactions;
    }

    public byte[] getMerkleRoot() {
        return merkleRoot;
    }

    public void setMerkleRoot(byte[] merkleRoot) {
        this.merkleRoot = merkleRoot;
    }

    public long getTimestamp() {
        return timestamp;
    }

    public void setTimestamp(long timestamp) {
        this.timestamp = timestamp;
    }

    /**
     * Calculates the hash using relevant fields of this type
     * @return SHA256-hash as raw bytes
     */
    public byte[] calculateHash() {
    	return CryptoUtils.hashObject(new Object[] {previousBlockHash, merkleRoot, timestamp, miningNonce} );
    }

    /**
     * Calculates the Hash of all transactions as hash tree.
     * https://en.wikipedia.org/wiki/Merkle_tree
     * @return SHA256-hash as raw bytes
     */
    public byte[] calculateMerkleRoot() {
    	Queue<byte[]> hashQueue = new LinkedList<byte[]>();
    	for (Transaction t : transactions) {
    		hashQueue.add(t.getHash());
    	}
        while (hashQueue.size() > 1) {
            // take 2 hashes from queue
            Object[] hashableData = new Object[] { hashQueue.poll(), hashQueue.poll() } ;
            // put new hash at end of queue
            hashQueue.add(CryptoUtils.hashObject(hashableData));
        }
        return hashQueue.poll();
    }

    public int getLeadingZerosCount() {
    	byte[] h = getHash();
        for (int i = 0; i < h.length; i++) {
            if (h[i] != 0) {
                return i;
            }
        }
        return h.length;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        Block block = (Block) o;
        return hash.equals(block.hash);
    }

    @Override
    public int hashCode() {
        return Objects.hash(previousBlockHash, merkleRoot, timestamp, miningNonce);
    }

	@Override
	public String toString() {
		String s = new String(" Hash ").concat(Base64.getEncoder().encodeToString(this.hash));
		if (this.previousBlockHash != null) {
			s = s.concat(" Prev ").concat(Base64.getEncoder().encodeToString(this.previousBlockHash));
		} else {
			s = s.concat(" Genesis ");
		}
		for (Transaction t : this.transactions) {
			s = s.concat("\n").concat(t.toString());
		}
		s.concat("\n");
		return s;
	}
	
}
