package ca.evermann.joerg.blockchainWFMS.chain;

import java.io.Serializable;
import java.security.PrivateKey;
import java.util.Objects;
import java.util.UUID;

import ca.evermann.joerg.blockchainWFMS.CA.CryptoUtils;
import ca.evermann.joerg.blockchainWFMS.CA.PeerCertificate;

public abstract class Transaction implements Serializable, Comparable<Transaction> {

	protected final UUID uuid = UUID.randomUUID();
	protected PeerCertificate originator;
	private byte[] signature;
	protected long timestamp;
	public Serializable payload;

	public Transaction(Serializable payload) {
		this.timestamp = System.currentTimeMillis();
		this.payload = payload;
	}
	
	public void setOriginator(PeerCertificate originator) { this.originator = originator; }
	public PeerCertificate getOriginator() { return originator; }
	public boolean verifyOriginator() {
		if (originator == null) {
			return false;
		}
		return originator.verifySignature();
	}
	public void signContent(PrivateKey key) {
		signature = CryptoUtils.signObject(new Object[] {originator, payload, timestamp}, key);
	}
	public byte[] getSignature() {
		return signature;
	}
	public boolean verifyContent() {
		if (payload == null || signature == null || originator == null) {
			return false;
		}
		return CryptoUtils.verifyObjectSignature( new Object[] {originator, payload, timestamp}, signature, originator.getKey());
	}
	public UUID getID() {
		return uuid;
	}
	
	public long getTimestamp() {
	    return timestamp;
	}
	
	/* This hash is for crypto signatures */
	public byte[] getHash() {
		return CryptoUtils.hashObject(new Object[] {originator, payload, timestamp} );
	}
	
	/* This hash is for Java hashed collections */
	@Override
	public int hashCode() {
		return Objects.hash(uuid);
	}
	
	@Override
	public boolean equals(Object obj) {
		if (obj == null) return false;
		if (this == obj) return true;
		if (getClass() != obj.getClass()) return false;
		return this.hashCode() == obj.hashCode();
	}
	
	@Override
	public String toString() {
		String s = new String("                " + this.getClass().getSimpleName() + " UUID ").concat(uuid.toString()).concat(" timestamp ").concat(Long.toString(timestamp));
		if (originator != null && payload != null) {
			s = s.concat(" originator ").concat(originator.toString()).concat(" payload ").concat(this.payload.toString());
		}
		return s;
	}

	@Override
	public int compareTo(Transaction o) {
		if (this.timestamp < o.timestamp) return -1;
		if (this.timestamp > o.timestamp) return 1;
		return 0;
	}

}
