package ca.evermann.joerg.blockchainWFMS.chain;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.HashSet;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;
import java.util.Vector;

import ca.evermann.joerg.blockchainWFMS.CA.PeerCertificate;
import ca.evermann.joerg.blockchainWFMS.p2p.P2PNode;
import ca.evermann.joerg.blockchainWFMS.p2p.messages.TransactionPoolSendMessage;
import ca.evermann.joerg.blockchainWFMS.p2p.messages.TransactionSendMessage;

public class TransactionService {

	private HashSet<Transaction> transactionPool = new HashSet<Transaction>();
	private List<Transaction> orphanPool = new Vector<Transaction>();

	protected P2PNode p2pnode;

	public TransactionService(P2PNode node) {
		this.p2pnode = node;
		this.p2pnode.transactionService = this;
		/*
		 * Transaction pools are meant to be transient, but for testing purposes we can
		 * persist and recover them TODO: Remove when done testing
		 */
		readTransactionPool();
	}

	/*
	 * Transaction pools are meant to be transient, but for testing purposes we can
	 * persist and recover them TODO: Remove when done testing
	 */
	public synchronized void persistTransactionPool() {
		ObjectOutputStream o;
		try {
			o = new ObjectOutputStream(new FileOutputStream(
					p2pnode.whoAmI().getHost() + "." + p2pnode.whoAmI().getPort() + ".transactionPool.js"));
			o.writeObject(this.transactionPool);
			o.writeObject(this.orphanPool);
			o.close();
		} catch (IOException e) {
			System.err.println("Error while saving transaction pool");
		}
	}

	@SuppressWarnings("unchecked")
	private synchronized void readTransactionPool() {
		ObjectInputStream o = null;
		try {
			o = new ObjectInputStream(new FileInputStream(
					p2pnode.whoAmI().getHost() + "." + p2pnode.whoAmI().getPort() + ".transactionPool.js"));
			this.transactionPool = (HashSet<Transaction>) o.readObject();
			this.orphanPool = (Vector<Transaction>) o.readObject();
			o.close();
			System.err.println(this.toString());
		} catch (IOException e) {
			System.err.println("Error while loading transaction pool");
		} catch (ClassNotFoundException e) {
			System.err.println("Class not found while loading transaction pool");
		}
	}

	/*
	 * The mining service needs access to the transaction pool
	 */
	Set<Transaction> getTransactionPool() {
		return transactionPool;
	}

	/*
	 * This is the normal add that should be called for newly received transactions
	 */
	public synchronized boolean add(Transaction transaction, PeerCertificate fromPeer) {
		if (!this.p2pnode.blockService.isTransactionInMainChain(transaction)) {
			boolean result = addNoChainDupsCheck(transaction, fromPeer);
			/*
			 * If we've added a new transaction to the pool, check the orphan transactions
			 * to see if they can be added now as well
			 */
			if (result)
				checkOrphans();
			return result;
		}
		return false;
	}

	/*
	 * Bulk add from multiple blocks during chain reorg. Because these are dumped in
	 * from multiple blocks in no particular order, we need to re-verify them
	 * individually. Orphan transactions refer to this particular holding pool
	 */
	protected synchronized void bulkAddFromMultipleBlocks(List<Transaction> list) {
		orphanPool.addAll(list);
	}

	/*
	 * This is used by the block service when reorganizing the chain It omits the
	 * check for duplicates in the main chain (which is done by the add method
	 * above)
	 */
	protected synchronized boolean addNoChainDupsCheck(Transaction transaction, PeerCertificate fromPeer) {
		if (transaction.verifyOriginator() && transaction.verifyContent()) {
			if (this.p2pnode.workflowEngine.validateTransaction(transaction, transactionPool)) {
				if (transactionPool.add(transaction)) {
					/*
					 * Persist the pool
					 */
					persistTransactionPool();
					/*
					 * And show it as pending
					 */
					p2pnode.workflowEngine.addPendingTransaction(transaction);
					/*
					 * Notify peers about a new transaction
					 */
					p2pnode.sendMessageExceptTo(new TransactionSendMessage(transaction), fromPeer);
					System.out.println("Added transaction to pool, new pool:");
					System.out.println(this.toString());
					return true;
				}
			}
		}
		return false;
	}

	synchronized void checkOrphans() {
		/*
		 * Ideally we'd do this recursively, but we'd end up with multiple concurrent
		 * iterators and get thrown a ConcurrentModificationException
		 */
		boolean moved;
		do {
			moved = false;
			ListIterator<Transaction> iter = orphanPool.listIterator();
			while (iter.hasNext()) {
				Transaction tx = iter.next();
				if (addNoChainDupsCheck(tx, null)) {
					iter.remove();
					moved = true;
				}
			}
		} while (moved);
	}

	synchronized void remove(List<Transaction> list) {
		transactionPool.removeAll(list);
		orphanPool.removeAll(list);
	}

	public synchronized void sendTransactionPoolTo(PeerCertificate recipient) {
		if (transactionPool.size() > 0) {
			p2pnode.sendMessageTo(recipient, new TransactionPoolSendMessage(transactionPool));
		}
	}

	@Override
	public String toString() {
		String s = new String("Transaction Pool\n");
		for (Transaction t : this.transactionPool) {
			s = s.concat(t.toString()).concat("\n");
		}
		s = s.concat("Orphan Pool\n");
		for (Transaction t : this.orphanPool) {
			s = s.concat(t.toString()).concat("\n");
		}
		return s;
	}
}