package ca.evermann.joerg.blockchainWFMS.workflow;

import java.io.File;
import java.io.PrintWriter;
import java.io.Serializable;
import java.io.StringWriter;
import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Arrays;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;
import java.util.Map.Entry;
import java.util.UUID;
import java.util.Vector;

import ca.evermann.joerg.blockchainWFMS.PetriNet.Transition;
import ca.evermann.joerg.blockchainWFMS.main.BlockChainUtils;

import javax.tools.DiagnosticCollector;
import javax.tools.JavaCompiler;
import javax.tools.JavaCompiler.CompilationTask;
import javax.tools.JavaFileObject;
import javax.tools.SimpleJavaFileObject;
import javax.tools.ToolProvider;

public class ActivityInstance implements Serializable {

	private 			String				pnName;
	private 			UUID				caseId;
	private 			Transition			t;

	private TreeMap<String, Serializable>	inputValues;
	private TreeMap<String, Serializable>	preValues;
	private TreeMap<String, Serializable>	postValues;

	private class JavaSourceFromString extends SimpleJavaFileObject {
		  final String code;

		  JavaSourceFromString(String name, String code) {
		    super(URI.create("string:///" + name.replace('.','/') + Kind.SOURCE.extension),Kind.SOURCE);
		    this.code = code;
		  }

		  @Override
		  public CharSequence getCharContent(boolean ignoreEncodingErrors) {
		    return code;
		  }
		}	
	
	public ActivityInstance(String pnName, UUID caseId, Transition t) {
		this.pnName = pnName;
		this.caseId = caseId;
		this.t = t;
		this.inputValues = null;
		this.preValues = null;
		this.postValues = null;
	}
	
	protected Transition getTransition() {
		return t;
	}
	
	protected String getPNName() {
		return pnName;
	}
	
	protected UUID getCaseId() {
		return caseId;
	}
	
	protected void setInputValues(PetriNetInstance pnInstance) {
		if (pnInstance == null) {
			System.err.println("ActivityInstance ("+t.getName()+"/"+pnName+") not connected to PetriNetInstance");
			return;
		}
		/* populate the input fields */
		inputValues = new TreeMap<String, Serializable>();
		for (String inName : t.getActivity().getInputs()) {
			inputValues.put(inName, BlockChainUtils.deepCopy(pnInstance.getValues().get(inName)));
		}
		/* save the pre-values of the outputs for undo */
		preValues = new TreeMap<String, Serializable>();
		for (String outName : t.getActivity().getOutputs()) {
			preValues.put(outName,  BlockChainUtils.deepCopy(pnInstance.getValues().get(outName)));
		}
	}
	
	protected Map<String, Serializable> getInputValues() {
		return inputValues;
	}
	
	protected void setOutputValues(Map<String, Serializable> outputs) {
		/* save them with deep copy */
		this.postValues = new TreeMap<String, Serializable>();
		for (Entry<String, Serializable> e : outputs.entrySet()) {
			postValues.put(e.getKey(), BlockChainUtils.deepCopy(e.getValue()));
		}
	}

	protected Map<String, Serializable> getPreValues() {
		return preValues;
	}

	protected Map<String, Serializable> getPostValues() {
		return postValues;
	}

	protected boolean undo(PetriNetInstance pnInstance) {
		if (preValues == null) {
			/* don't have pre-values, cannot undo */
			System.err.println("ActivityInstance ("+t.getName()+"/"+pnName+") has no pre-values for undo");
			return false;
		}
		if (pnInstance == null) {
			System.err.println("ActivityInstance ("+t.getName()+"/"+pnName+") not connected to PetriNetInstance");
			return false;
		}
		/* write the pre-values to the instance */
		for (String outName : preValues.keySet()) {
			pnInstance.getValues().put(outName, BlockChainUtils.deepCopy(preValues.get(outName)));
		}
		/* and update the instance marking */
		pnInstance.unfire(getTransition());
		return true;
	}
	
	protected boolean execute(PetriNetInstance pnInstance) {
		if (postValues == null) {
			/* don't have post-values, cannot re */
			System.err.println("ActivityInstance ("+t.getName()+"/"+pnName+") has no post-values for execute");
			return false;
		}
		if (pnInstance == null) {
			System.err.println("ActivityInstance ("+t.getName()+"/"+pnName+") not connected to PetriNetInstance");
			return false;
		}
		/* write the post-values to the PN instance */
		for (String outName : postValues.keySet()) {
			pnInstance.getValues().put(outName, BlockChainUtils.deepCopy(postValues.get(outName)));
		}
		/* And update the instance marking */
		pnInstance.fire(getTransition());
		return true;
	}
	
	@Override
	public int hashCode() {
		return Objects.hash(caseId, pnName, t.getName());
	}
	
	@Override
	public boolean equals(Object obj) {
		if (obj == null) return false;
		if (this == obj) return true;
		if (getClass() != obj.getClass()) return false;
		return this.hashCode() == obj.hashCode();
	}
	
	@Override
	public String toString() {
		return new String("Instance of " + t.getName() + " for PN " + pnName + " for case " + caseId.toString());
	}

	@SuppressWarnings("rawtypes")
	public boolean checkConstraints() {
		if (!getTransition().getActivity().getConstraints().isEmpty()) {
			JavaCompiler compiler = ToolProvider.getSystemJavaCompiler();
			DiagnosticCollector<JavaFileObject> diagnostics = new DiagnosticCollector<JavaFileObject>();
	
			StringWriter writer = new StringWriter();
			PrintWriter out = new PrintWriter(writer);
			out.println("import java.io.Serializable;");
			for (Class cl : getTransition().getActivity().getTypes().values()) {
				out.println("import "+cl.getName()+";");
			}
			out.println("public class ConstraintEval {");
			out.print("  public static boolean eval(");
			Iterator<String> iter = postValues.keySet().iterator();
			while(iter.hasNext()) {
				String name = iter.next();
				out.print(getTransition().getActivity().getTypes().get(name).getName() + " " + name);
				if (iter.hasNext()) {
					out.print(", ");
				}
			}
			out.println(") {");
			out.println("    boolean r = true;");
			for (String c : getTransition().getActivity().getConstraints()) {
				out.println("    r = r && ("+c+");");
			}
			out.println("    return r;");
			out.println("  }");
			out.println("}");
			out.close();
			JavaFileObject file = new JavaSourceFromString("ConstraintEval", writer.toString());
	
			Iterable<? extends JavaFileObject> compilationUnits = Arrays.asList(file);
			CompilationTask task = compiler.getTask(null, null, diagnostics, null, null, compilationUnits);		    
			boolean success = task.call();
			if (success) {
				try {
					Vector<Class> inputClasses = new Vector<Class>();
					iter = postValues.keySet().iterator();
					while(iter.hasNext()) {
						String name = iter.next();
						inputClasses.add(getTransition().getActivity().getTypes().get(name));
					}
		            URLClassLoader classLoader = URLClassLoader.newInstance(new URL[] { new File("").toURI().toURL() });
					Boolean r = (Boolean) Class.forName("ConstraintEval", true, classLoader).getDeclaredMethod("eval", inputClasses.toArray(new Class[] {})).invoke(null, postValues.values().toArray());
					return r;
				} catch (ClassNotFoundException e) {
					System.err.println("Class not found: " + e);
				} catch (NoSuchMethodException e) {
					System.err.println("No such method: " + e);
				} catch (IllegalAccessException e) {
					System.err.println("Illegal access: " + e);
				} catch (InvocationTargetException e) {
					System.err.println("Invocation target: " + e);
				} catch (MalformedURLException e) {
					System.err.println("Malformed URL: " + e);
				}
				return false;
			}
			return false;
		}
		return true;
	}

	@SuppressWarnings("rawtypes")
	public Map<String, Serializable> callExecute() {
		JavaCompiler compiler = ToolProvider.getSystemJavaCompiler();
		DiagnosticCollector<JavaFileObject> diagnostics = new DiagnosticCollector<JavaFileObject>();

		StringWriter writer = new StringWriter();
		PrintWriter out = new PrintWriter(writer);
		out.println("import java.io.Serializable;");
		for (Class cl : getTransition().getActivity().getTypes().values()) {
			out.println("import "+cl.getName()+";");
		}
		out.println("import "+getTransition().getActivity().getExecClass()+";");
		out.println("public class CallExecute {");
		out.print("  public static Serializable exec(");
		Iterator<String> iter = inputValues.keySet().iterator();
		while(iter.hasNext()) {
			String name = iter.next();
			out.print(getTransition().getActivity().getTypes().get(name).getName() + " " + name);
			if (iter.hasNext()) {
				out.print(", ");
			}
		}
		out.println(") {");
		out.print("    Serializable r = "+getTransition().getActivity().getExecClass()+"."+getTransition().getActivity().getExecMethod()+"(");
		iter = inputValues.keySet().iterator();
		while(iter.hasNext()) {
			String name = iter.next();
			out.print(name);
			if (iter.hasNext()) {
				out.print(", ");
			}
		}
		out.println(");");
		out.println("    return r;");
		out.println("  }");
		out.println("}");
		out.close();
		JavaFileObject file = new JavaSourceFromString("CallExecute", writer.toString());

		Iterable<? extends JavaFileObject> compilationUnits = Arrays.asList(file);
		CompilationTask task = compiler.getTask(null, null, diagnostics, null, null, compilationUnits);		    
		boolean success = task.call();
		TreeMap<String, Serializable> r = new TreeMap<String, Serializable>();
		if (success) {
			try {
				Vector<Class> inputClasses = new Vector<Class>();
				iter = inputValues.keySet().iterator();
				while(iter.hasNext()) {
					String name = iter.next();
					inputClasses.add(getTransition().getActivity().getTypes().get(name));
				}
	            URLClassLoader classLoader = URLClassLoader.newInstance(new URL[] { new File("").toURI().toURL() });
				Serializable obj = (Serializable) Class.forName("CallExecute", true, classLoader).getDeclaredMethod("exec", inputClasses.toArray(new Class[] {})).invoke(null, inputValues.values().toArray() );
				r.put(getTransition().getActivity().getOutputs().toArray(new String[] {})[0], obj);
				return r;
			} catch (ClassNotFoundException e) {
				System.err.println("Class not found: " + e);
			} catch (NoSuchMethodException e) {
				System.err.println("No such method: " + e);
			} catch (IllegalAccessException e) {
				System.err.println("Illegal access: " + e);
			} catch (InvocationTargetException e) {
				System.err.println("Invocation target: " + e);
			} catch (MalformedURLException e) {
				System.err.println("Malformed URL: " + e);
			}
		}
		return r;
	}

}
