package ca.evermann.joerg.blockchainWFMS.workflow;

import java.io.Serializable;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;

import ca.evermann.joerg.blockchainWFMS.PetriNet.PetriNet;
import ca.evermann.joerg.blockchainWFMS.PetriNet.Place;
import ca.evermann.joerg.blockchainWFMS.PetriNet.Transition;
import ca.evermann.joerg.blockchainWFMS.main.BlockChainUtils;

public class PetriNetInstance implements Serializable {

	private HashMap<Place, Integer>			marking;
	private PetriNet						net;
	private UUID							id;
	private HashMap<String, Serializable> 	values; 
	
	public PetriNetInstance(PetriNet net, UUID caseID) {
		this.id = caseID;
		this.net = net;
		this.marking = new HashMap<Place, Integer>();
		this.values = new HashMap<String, Serializable>();
		
		for (Place p : this.net.getPlaces()) {
			marking.put(p,  0);
		}
		marking.put(net.getSource(), 1);
	}
	
	public Set<Transition> getEnabled() {
		Set<Transition> enabledSet = new HashSet<Transition>();
		for (Transition t : net.getTransitions()) {
			boolean e = true;
			Set<Place> preset = net.getPreset(t);
			for (Place p : preset) {
				if (marking.get(p) < 1) {
					e = false;
					break;
				}
			}
			if (e == true) {
				enabledSet.add(t);
			}
		}
		return enabledSet;
	}
	
	public UUID getId() {
		return id;
	}
	
	public PetriNet	getNet() {
		return net;
	}
	
	public HashMap<String, Serializable> getValues() {
		return values;
	}
	
	public boolean isDeadlocked() {
		return (getEnabled().size()==0 && !isDone());
	}
	
	public void fire(Transition t) {
		// check if this is enabled
		boolean e = true;
		for (Place p : net.getPreset(t)) {
			if (marking.get(p) < 1) {
				e = false;
				break;
			}
		}
		// update marking
		if (e == true) {
			for (Place p : net.getPreset(t)) {
				marking.put(p, marking.get(p)-1);
			}
			for (Place p : net.getPostset(t)) {
				marking.put(p, marking.get(p)+1);
			}
		}
	}
	
	public void unfire(Transition t) {
		/*
		 *  check if this is possible 
		 *  (all post-places must have one or more tokens)
		 */
		boolean e = true;
		for (Place p : net.getPostset(t)) {
			if (marking.get(p) < 1) {
				e = false;
				break;
			}
		}
		if (e == true) {
			// update marking
			for (Place p : net.getPreset(t)) {
				marking.put(p, marking.get(p)+1);
			}
			for (Place p : net.getPostset(t)) {
				marking.put(p, marking.get(p)-1);
			}
			// check if t is now enabled
			boolean enabled = true;
			for (Place p : net.getPreset(t)) {
				if (marking.get(p) < 1) {
					enabled = false;
					break;
				}
			}
			if (!enabled) {
				System.err.println("Unfired transition "+t.getName()+" for PNI "+getNet().getName()+"/"+getId()+" that was not enabled.");
			}
		} else {
			System.err.println("Could not Unfire transition "+t.getName()+" for PNI "+getNet().getName()+"/"+getId()+" that was not enabled.");
		}
	}
	
	public boolean isDone() {
		boolean d = true;
		for (Place p : net.getPlaces()) {
			if (p != net.getSink()) {
				if (marking.get(p) != 0) {
					d = false;
					break;
				}
			} else {
				if (marking.get(p) != 1) {
					d = false;
					break;
				}
			}
		}
		return d;
	}
	
	public boolean isInitial() {
		boolean d = true;
		for (Place p : net.getPlaces()) {
			if (p != net.getSource()) {
				if (marking.get(p) != 0) {
					d = false;
					break;
				}
			} else {
				if (marking.get(p) != 1) {
					d = false;
					break;
				}
			}
		}
		return d;
	}
	
	@Override
	public int hashCode() {
		return Objects.hash(net.getName(), id);
	}
	
	@Override
	public boolean equals(Object obj) {
		if (obj == null) return false;
		if (this == obj) return true;
		if (getClass() != obj.getClass()) return false;
		return this.hashCode() == obj.hashCode();
	}

	@Override
	public String toString() {
		String r = "PNI: " + net.getName() + "/" + BlockChainUtils.right(id.toString(), 4) + " (";
		for (Map.Entry<Place, Integer> e : marking.entrySet()) {
			r += BlockChainUtils.right(e.getKey().getId().toString(), 4) + "=" + e.getValue().toString()+ ", ";
		}
		r += ") (";
		for (Map.Entry<String, Serializable> e : values.entrySet()) {
			r += e.getKey() + "=" + e.getValue().toString()+ ", ";
		}
		r += ")";
		return r;
	}
}
