package ca.evermann.joerg.blockchainWFMS.workflow;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import ca.evermann.joerg.blockchainWFMS.PetriNet.ActivitySpec;
import ca.evermann.joerg.blockchainWFMS.PetriNet.PetriNet;
import ca.evermann.joerg.blockchainWFMS.PetriNet.Transition;
import ca.evermann.joerg.blockchainWFMS.chain.Transaction;
import ca.evermann.joerg.blockchainWFMS.workflow.transactions.FireTransitionTransaction;
import ca.evermann.joerg.blockchainWFMS.workflow.transactions.InitCaseTransaction;
import ca.evermann.joerg.blockchainWFMS.workflow.transactions.ModelUpdateTransaction;

public class VisibleWorkflowState extends WorkflowState {

	private Worklist 					worklist;
	private HashMap<Transaction, Integer> 	pendingTransactions;
	private WorklistUI					worklistUI;

	VisibleWorkflowState(WorkflowEngine engine, int lag) {
		super(engine, lag);
		worklist = new Worklist();
		pendingTransactions = new HashMap<Transaction, Integer>();
		worklistUI = new WorklistUI(engine.p2pnode);
		worklistUI.setWorklistAndPending(worklist, pendingTransactions);
		this.worklistUI.setVisible(true); 
	}

	@SuppressWarnings("unchecked")
	@Override
	public void readState(ObjectInputStream o) {
		super.readState(o);
		/*
		 * Also read the worklist
		 */
		try {
			worklist = (Worklist) o.readObject();
			pendingTransactions = (HashMap<Transaction, Integer>) o.readObject();
		} catch (IOException e) {
			System.err.println("Error while loading workflow state");
		} catch (ClassNotFoundException e) {
			System.err.println("Class not found while loading workflow state");
		}
		/*
		 * And update the GUI
		 */
		worklistUI.setWorklistAndPending(worklist, pendingTransactions);
		worklistUI.resetModels();
		for (PetriNet pn : knownPetriNets.values()) {
			worklistUI.addToWorkflowModel(pn);
		}
		for (PetriNetInstance pni : runningInstances.values()) {
			worklistUI.addToOrUpdateCaseModel(pni);
		}
		for (ActivityInstance actInstance : worklist.asList()) {
			worklistUI.addToWorklistModel(actInstance);
		}
		for (Transaction t : pendingTransactions.keySet()) {
			worklistUI.addToPendingTransactions(t, pendingTransactions.get(t));
		}
	}

	@Override
	public void persistState(ObjectOutputStream o) {
		super.persistState(o);
		/*
		 * Also write the worklist
		 */
		try {
			o.writeObject(worklist);
			o.writeObject(pendingTransactions);
		} catch (IOException e) {
			System.err.println("Error while saving workflow state");
		}
	}

	public void enableWorkflowUI() {
		this.worklistUI.setEnabled(true);
		this.worklistUI.setVisible(true); 
	}
	
	public void disableWorkflowUI() {
		this.worklistUI.setEnabled(false);
	}
	
    @Override
    protected void doTransaction(Transaction t) {
    	super.doTransaction(t);
		worklistUI.removeFromPendingTransactions(t);
    }
    
    @Override
    protected void undoTransaction(Transaction t) {
    	super.undoTransaction(t);
    }

    protected PetriNet doModelUpdateTransaction(ModelUpdateTransaction t) {
    	PetriNet pn = super.doModelUpdateTransaction(t);
    	if (pn != null) { 
    		worklistUI.addToWorkflowModel(pn);
    	}
		return pn;
	}

    protected PetriNet undoModelUpdateTransaction(ModelUpdateTransaction t) {
    	PetriNet pn = super.undoModelUpdateTransaction(t);
		if (pn != null) {
			worklistUI.removeFromWorkflowModel(pn);
		}
		return pn;
	}
    
    protected PetriNetInstance doInitCaseTransaction(InitCaseTransaction t) {
    	PetriNetInstance pni = super.doInitCaseTransaction(t);
    	if (pni != null) {
    		worklistUI.addToOrUpdateCaseModel(pni);
    		for (Transition transition : pni.getEnabled()) {
    			ActivitySpec activity = transition.getActivity();
    			if (activity.getHost().equals(engine.p2pnode.whoAmI().getHost()) && activity.getPort().equals(engine.p2pnode.whoAmI().getPort())) {
    				/*
    				 * Only if its destined for the local node, do we create an activity instance
    				 * either for the local worklist or to execute externally locally
    				 */
    				ActivityInstance actInstance = new ActivityInstance(pni.getNet().getName(), pni.getId(), transition);
    				actInstance.setInputValues(pni);	
    				if (activity.getExecClass().length()==0 || activity.getExecMethod().length()==0) {
    					// we don't have execution stuff specified, so we assume it's a manual work item, put it in the worklist
    					worklist.put(actInstance);
    					worklistUI.addToWorklistModel(actInstance);
    				} else {
    					Map<String, Serializable> outputs = actInstance.callExecute();
    					actInstance.setOutputValues(outputs);
    					Transaction tReply = new FireTransitionTransaction(actInstance);
    					tReply.setOriginator(engine.p2pnode.whoAmI());
    					engine.p2pnode.signTransaction(tReply);
    					if (!engine.p2pnode.transactionService.add(tReply, engine.p2pnode.whoAmI())) {
    						System.err.println("Automatic transition transaction could not be validated.\nThere may be a pending conflicting transaction.\nYou may retry later.");
    					}
    				}
    			}
    		}
    	}
    	return pni;
	}

    protected PetriNetInstance undoInitCaseTransaction(InitCaseTransaction t) {
    	PetriNetInstance pni = super.undoInitCaseTransaction(t);
    	if (pni != null) {
    		worklist.removeCase(pni.getId().toString());
    		worklistUI.removeCaseFromWorklistModel(pni.getId());
    		worklistUI.removeFromCaseModel(pni.getId());
    	}
    	return pni;
	}

    protected PetriNetInstance doFireTransitionTransaction(FireTransitionTransaction t) {
		ActivityInstance ativityInstance = (ActivityInstance)t.payload;
		PetriNetInstance pni = getPetriNetInstance(ativityInstance);
		/*
		 * Find the set of previously enabled transitions
		 */
		Set<Transition> previousEnabled = new HashSet<Transition>();
		if (pni != null) {
			/*
			 * Find the items in our local worklist that were enabled for this instance
			 */
    		for (ActivityInstance ai : worklist.get(pni.getId().toString())) {
    			previousEnabled.add(ai.getTransition());
    		}
		}
		/*
		 * Execute this transition firing transaction
		 */
		pni = super.doFireTransitionTransaction(t);
		/*
		 * And find out the newly enabled transitions
		 * to update the worklist
		 */
		Set<Transition> newlyEnabled = new HashSet<Transition>();
		if (pni != null) {
			/*
			 * Only transitions destined for the local node get considered here
			 */
			for (Transition txEnabled : pni.getEnabled()) {
				if (txEnabled.getActivity().getHost().equals(engine.p2pnode.whoAmI().getHost()) && txEnabled.getActivity().getPort().equals(engine.p2pnode.whoAmI().getPort())) {
					newlyEnabled.add(txEnabled);
				}
			}
		}
		
		Set<Transition> toRemove = new HashSet<Transition>(previousEnabled);
		toRemove.removeAll(newlyEnabled);
		Set<Transition> toAdd = new HashSet<Transition>(newlyEnabled);
		toAdd.removeAll(previousEnabled);

		for(Transition tx : toRemove) {
			worklistUI.removeFromWorklistModel(pni.getId().toString(), tx.getName());
			worklist.remove(pni.getId().toString(), tx.getName());
		}
		for(Transition tx : toAdd) {
			ActivitySpec activity = tx.getActivity();
			if (activity.getHost().equals(engine.p2pnode.whoAmI().getHost()) && activity.getPort().equals(engine.p2pnode.whoAmI().getPort())) {
				ActivityInstance actInstance = new ActivityInstance(pni.getNet().getName(), pni.getId(), tx);
				actInstance.setInputValues(pni);
				if (activity.getExecClass().length()==0 || activity.getExecMethod().length()==0) {
					// we don't have execution stuff specified, so we assume it's a manual work item, put it in the worklist
					worklist.put(actInstance);
					worklistUI.addToWorklistModel(actInstance);
				} else {
					Map<String, Serializable> outputs = actInstance.callExecute();
					actInstance.setOutputValues(outputs);
					Transaction tReply = new FireTransitionTransaction(actInstance);
					tReply.setOriginator(engine.p2pnode.whoAmI());
					engine.p2pnode.signTransaction(tReply);
					if (!engine.p2pnode.transactionService.add(tReply, engine.p2pnode.whoAmI())) {
						System.err.println("Automatic transition transaction could not be validated.\nThere may be a pending conflicting transaction.\nYou may retry later.");
					}
				}
			}
		}
		/*
		 * Update the case model
		 */
		worklistUI.addToOrUpdateCaseModel(pni);

    	return pni;
	}

    /* 
     * undoFireTransitionTransaction works exactly the same way as doFireTransitionTransaction
     * except that we call super.undoFireTransitionTransaction which undoes the activity instance, which
     * unfires the petri net instance marking
     * 
     * From a Petri net perspective there is no difference whether we fire forwards or backwards
     */
    protected PetriNetInstance undoFireTransitionTransaction(FireTransitionTransaction t) {
		ActivityInstance ativityInstance = (ActivityInstance)t.payload;
		PetriNetInstance pni = getPetriNetInstance(ativityInstance);
		/*
		 * Find the set of previously enabled transitions
		 */
		Set<Transition> previousEnabled = new HashSet<Transition>();
		if (pni != null) {
			/*
			 * Find the items in our local worklist that were enabled for this instance
			 */
    		for (ActivityInstance ai : worklist.get(pni.getId().toString())) {
    			previousEnabled.add(ai.getTransition());
    		}
		}
		/*
		 * Execute this transition firing transaction
		 */
		pni = super.undoFireTransitionTransaction(t);
		/*
		 * And find out the newly enabled transitions
		 * to update the worklist
		 */
		Set<Transition> newlyEnabled = new HashSet<Transition>();
		if (pni != null) {
			/*
			 * Only transitions destined for the local node get considered here
			 */
			for (Transition txEnabled : pni.getEnabled()) {
				if (txEnabled.getActivity().getHost().equals(engine.p2pnode.whoAmI().getHost()) && txEnabled.getActivity().getPort().equals(engine.p2pnode.whoAmI().getPort())) {
					newlyEnabled.add(txEnabled);
				}
			}
		}
		
		Set<Transition> toRemove = new HashSet<Transition>(previousEnabled);
		toRemove.removeAll(newlyEnabled);
		Set<Transition> toAdd = new HashSet<Transition>(newlyEnabled);
		toAdd.removeAll(previousEnabled);

		for(Transition tx : toRemove) {
			worklistUI.removeFromWorklistModel(pni.getId().toString(), tx.getName());
			worklist.remove(pni.getId().toString(), tx.getName());
		}
		for(Transition tx : toAdd) {
			ActivitySpec activity = tx.getActivity();
			if (activity.getHost().equals(engine.p2pnode.whoAmI().getHost()) && activity.getPort().equals(engine.p2pnode.whoAmI().getPort())) {
				ActivityInstance actInstance = new ActivityInstance(pni.getNet().getName(), pni.getId(), tx);
				actInstance.setInputValues(pni);
				worklist.put(actInstance);
				worklistUI.addToWorklistModel(actInstance);
			}
		}
		/*
		 * Update the case model
		 */
		worklistUI.addToOrUpdateCaseModel(pni);

    	return pni;
	}

	public void addPendingTransaction(Transaction transaction) {
		switch (transaction.getClass().getName()) {
		case "ca.evermann.joerg.blockchainWFMS.workflow.transactions.FireTransitionTransaction":
		case "ca.evermann.joerg.blockchainWFMS.workflow.transactions.InitCaseTransaction":
		case "ca.evermann.joerg.blockchainWFMS.workflow.transactions.ModelUpdateTransaction":
			worklistUI.addToPendingTransactions(transaction, -1);
			break;
		default:
			break;
		}
	}

	@Override
	void handlePendingTransaction(Transaction t, int depth) {
		switch (t.getClass().getName()) {
		case "ca.evermann.joerg.blockchainWFMS.workflow.transactions.FireTransitionTransaction":
		case "ca.evermann.joerg.blockchainWFMS.workflow.transactions.InitCaseTransaction":
		case "ca.evermann.joerg.blockchainWFMS.workflow.transactions.ModelUpdateTransaction":
			worklistUI.addToPendingTransactions(t, depth);
			break;
		default:
			break;
		}
	}

    @Override
    public String toString() {
    	return super.toString() + worklist.toString();
    }

}