package ca.evermann.joerg.blockchainWFMS.workflow;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import ca.evermann.joerg.blockchainWFMS.PetriNet.PetriNet;
import ca.evermann.joerg.blockchainWFMS.chain.Block;
import ca.evermann.joerg.blockchainWFMS.chain.Transaction;
import ca.evermann.joerg.blockchainWFMS.workflow.transactions.FireTransitionTransaction;
import ca.evermann.joerg.blockchainWFMS.workflow.transactions.InitCaseTransaction;
import ca.evermann.joerg.blockchainWFMS.workflow.transactions.ModelUpdateTransaction;

public abstract class WorkflowState implements Serializable {
	
	protected Map<UUID, PetriNetInstance> 	runningInstances;
	protected Map<String, PetriNet>			knownPetriNets;
	protected WorkflowEngine				engine;

	protected final int						lag;
	
	WorkflowState(WorkflowEngine engine, int lag) {
		this.engine = engine;
		this.lag = lag;
		
		knownPetriNets = new HashMap<String, PetriNet>();
		runningInstances = new HashMap<UUID, PetriNetInstance>();
	}
	
    @SuppressWarnings("unchecked")
	void readState(ObjectInputStream o) {
		try {
			this.runningInstances = (HashMap<UUID, PetriNetInstance>) o.readObject();
			this.knownPetriNets = (HashMap<String, PetriNet>) o.readObject();
		} catch (IOException e) {
			System.err.println("Error while loading workflow state");
		} catch (ClassNotFoundException e) {
			System.err.println("Class not found while loading workflow state");
		}
    }
    
    void persistState(ObjectOutputStream o) {
		try {
			o.writeObject(this.runningInstances);
			o.writeObject(this.knownPetriNets);
		} catch (IOException e) {
			System.err.println("Error while saving workflow state");
		}
    }

	protected PetriNetInstance getPetriNetInstance(PetriNet pn, UUID caseId) {
		PetriNetInstance pni = runningInstances.get(caseId);
		if (!pni.getNet().equals(pn)) {
			System.err.println("Wrong PN for caseId "+caseId.toString()+". Expected+"+pn.getName()+", got "+pni.getNet().getName());
		}
		return pni;
	}
	
	protected PetriNet getPetriNet(String pnName) {
		return knownPetriNets.get(pnName);
	}
	
    void doBlock(Block b) {
    	for (int i=0; i<lag && b != null; i++) {
    	   	for (Transaction t : b.getTransactions()) {
        		handlePendingTransaction(t, i);
        	}
   			b = engine.p2pnode.blockService.getPredecessor(b);
    	}
    	if (b != null) {
    	   	for (Transaction t : b.getTransactions()) {
        		doTransaction(t);
        	}
    	}
    }
    
    void handlePendingTransaction(Transaction t, int depth) {}

	void undoBlock(Block b) {
    	for (int i=0; i<lag && b != null; i++) {
   			b = engine.p2pnode.blockService.getPredecessor(b);
    	}
    	if (b != null) {
    	   	for (Transaction t : b.getTransactions()) {
        		undoTransaction(t);
        	}
    	}
    }
    
    protected PetriNetInstance getPetriNetInstance(ActivityInstance ai) {
    	return runningInstances.get(ai.getCaseId());
    }

    protected void doTransaction(Transaction t) {
		System.err.println("   DO " + t.toString());
		switch (t.getClass().getName()) {
		case "ca.evermann.joerg.blockchainWFMS.chain.TestTransaction":
			break;
		case "ca.evermann.joerg.blockchainWFMS.workflow.transactions.FireTransitionTransaction":
			doFireTransitionTransaction( (FireTransitionTransaction)t );
			break;
		case "ca.evermann.joerg.blockchainWFMS.workflow.transactions.InitCaseTransaction":
			doInitCaseTransaction( (InitCaseTransaction)t );
			break;
		case "ca.evermann.joerg.blockchainWFMS.workflow.transactions.ModelUpdateTransaction":
			doModelUpdateTransaction( (ModelUpdateTransaction)t );
			break;
		default:
			break;
		}
	}
	
	protected void undoTransaction(Transaction t) {
		System.err.println("  UNDO" + t.toString());
		switch (t.getClass().getName()) {
		case "ca.evermann.joerg.blockchainWFMS.chain.TestTransaction":
			break;
		case "ca.evermann.joerg.blockchainWFMS.workflow.transactions.FireTransitionTransaction":
			undoFireTransitionTransaction( (FireTransitionTransaction)t );
			break;
		case "ca.evermann.joerg.blockchainWFMS.workflow.transactions.InitCaseTransaction":
			undoInitCaseTransaction( (InitCaseTransaction)t );
			break;
		case "ca.evermann.joerg.blockchainWFMS.workflow.transactions.ModelUpdateTransaction":
			undoModelUpdateTransaction( (ModelUpdateTransaction)t );
			break;
		default:
			break;
		}
	}

	protected PetriNet doModelUpdateTransaction(ModelUpdateTransaction t) {
		PetriNet pn = (PetriNet) t.payload;
		/* and we put it into the known nets, overwriting any earlier version */
		if (this.knownPetriNets.put(pn.getName(), pn) != null) {
			System.err.println("Added PN "+pn.getName()+" already existed, shouldn't have verified");
			return null;
		};
		return pn;
	}

	protected PetriNet undoModelUpdateTransaction(ModelUpdateTransaction t) {
		PetriNet pn = (PetriNet) t.payload;
		boolean instance = false;
		for (PetriNetInstance pni : runningInstances.values()) {
			if (pni.getNet().equals(pn)) {
				instance = true;
				break;
			}
		}
		if (instance) {
			System.err.println("Undo model update has running instances");
			return null;
		}
		knownPetriNets.remove(pn.getName());
		return pn;
	}

	protected PetriNetInstance doInitCaseTransaction(InitCaseTransaction t) {
		InitCaseTransaction.NetNameAndCaseId netNameAndCaseId = (InitCaseTransaction.NetNameAndCaseId) t.payload;
		PetriNet pn = this.knownPetriNets.get( netNameAndCaseId.getName() );
		if (pn == null) {
			System.err.println("Case init called for unknown PetriNet "+ netNameAndCaseId );
			return null;
		}
		PetriNetInstance pni = new PetriNetInstance(pn, netNameAndCaseId.getCaseId());
		runningInstances.put(netNameAndCaseId.getCaseId(), pni);
		return pni;
	}

	protected PetriNetInstance undoInitCaseTransaction(InitCaseTransaction t) {
		InitCaseTransaction.NetNameAndCaseId netNameAndCaseId = (InitCaseTransaction.NetNameAndCaseId) t.payload;
		PetriNet pn = this.knownPetriNets.get( netNameAndCaseId.getName() );
		if (pn == null) {
			System.err.println("Case init undo called for unknown PetriNet "+ netNameAndCaseId.getName() );
			return null;
		}
		PetriNetInstance pni = this.getPetriNetInstance(pn, netNameAndCaseId.getCaseId());
		if (pni == null) {
			System.err.println("Case init undo called for unknown PetriNetInstance: "+ netNameAndCaseId );
			return null;
		}
		if (!pni.isInitial()) {
			System.err.println("Case init undo called when PetriNetInstance is not initial state");
			return null;
		}
		runningInstances.remove(netNameAndCaseId.getCaseId());
		return pni;
	}

	protected PetriNetInstance doFireTransitionTransaction(FireTransitionTransaction t) {
		ActivityInstance activityInstance = (ActivityInstance)t.payload;
		PetriNetInstance pni = getPetriNetInstance(activityInstance);
		activityInstance.execute(pni);
		return pni;
	}

	protected PetriNetInstance undoFireTransitionTransaction(FireTransitionTransaction t) {
		ActivityInstance activityInstance = (ActivityInstance)t.payload;
		PetriNetInstance pni = getPetriNetInstance(activityInstance);
		activityInstance.undo(pni);
		return pni;
	}

	@Override
	public String toString() {
		String s = new String("  Running Instances:\n");
		for (PetriNetInstance pni : runningInstances.values()) {
			s = s.concat("    " + pni.toString() + "\n");
		}
		s = s.concat("  Known Petri nets:\n");
		for (String st : knownPetriNets.keySet()) {
			s = s.concat("    " + st + "\n");			
		}
		return s;
	}
}
