package ca.evermann.joerg.blockchainWFMS.workflow;

import java.awt.Container;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Image;
import java.awt.Label;
import java.awt.TextField;
import java.awt.event.ActionEvent;
import java.io.IOException;
import java.io.Serializable;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.Map.Entry;
import java.util.TreeMap;
import java.util.Vector;

import javax.imageio.ImageIO;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.filechooser.FileFilter;
import javax.swing.filechooser.FileNameExtensionFilter;
import javax.swing.table.DefaultTableModel;
import ca.evermann.joerg.blockchainWFMS.PetriNet.ActivitySpec;
import ca.evermann.joerg.blockchainWFMS.PetriNet.PetriNet;
import ca.evermann.joerg.blockchainWFMS.chain.Transaction;
import ca.evermann.joerg.blockchainWFMS.main.BlockChainWFMSConfig;
import ca.evermann.joerg.blockchainWFMS.main.BlockChainWFMSUI;
import ca.evermann.joerg.blockchainWFMS.p2p.P2PNode;
import ca.evermann.joerg.blockchainWFMS.workflow.transactions.FireTransitionTransaction;
import ca.evermann.joerg.blockchainWFMS.workflow.transactions.InitCaseTransaction;
import ca.evermann.joerg.blockchainWFMS.workflow.transactions.ModelUpdateTransaction;

public class WorklistUI extends JFrame {

	private		Worklist				worklist;
	private 	DefaultTableModel 		worklistModel;
	private 	JTable 					worklistTable;
	private 	JButton					workItemButton;
	private		DefaultTableModel		petriNetModel;
	private		JTable					petriNetTable;
	private		JButton					pnLoadButton;
	private		JButton					pnSaveButton;
	private		DefaultTableModel		caseModel;
	private		JTable					caseTable;
	private		JButton					caseButton;
	private		HashMap<Transaction, Integer>		pendingTransactions;
	private		DefaultTableModel		pendingTxModel;
	private		JTable					pendingTxTable;
	private JButton viewPendingButton;

	protected void setWorklistAndPending(Worklist worklist, HashMap<Transaction, Integer> pendingTransactions) {
		this.worklist = worklist;
		this.pendingTransactions = pendingTransactions;
	}
	
	protected WorklistUI(P2PNode p2pnode) {
		super();
		
		this.setTitle("WorklistUI for host " + p2pnode.whoAmI().getHost() + " on port " + p2pnode.whoAmI().getPort().toString());
		this.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
		this.setSize(600, 800);
		this.setMinimumSize(new Dimension(600, 800));

		Image icon;
		try {
			URL u = BlockChainWFMSUI.class.getResource("icons8-arbeitsablauf-80.png");
			icon = ImageIO.read(u);
			this.setIconImage(icon);
		} catch (IOException e) {
			// proceed without custom icon
		}

		workItemButton = new JButton("Start & Finish Work Item");
		workItemButton.addActionListener((ActionEvent event) -> {
			int selection = worklistTable.getSelectedRow();
			if (selection != -1) {
				ActivityInstance actInstance = worklist.get((String)worklistTable.getValueAt(selection, 1), (String)worklistTable.getValueAt(selection, 2));
				if (actInstance != null) {
					Map<String, Serializable> outputs = doWorkItem(actInstance);
					if (outputs != null) {
						actInstance.setOutputValues(outputs);
						Transaction t = new FireTransitionTransaction(actInstance);
						t.setOriginator(p2pnode.whoAmI());
						p2pnode.signTransaction(t);
						if (!p2pnode.transactionService.add(t, p2pnode.whoAmI())) {
							JOptionPane.showMessageDialog(this, "Transaction could not be validated.\nThere may be a pending conflicting transaction.\nYou may retry later.", "Transaction Error (Failure)", JOptionPane.ERROR_MESSAGE);
							/*
							 * Leave in worklist until the visible state has changed so it's no longer executable
							 */
						} else {
							/*
							 * If successful, we remove from the worklist, so user cannot redo it 
							 */
							worklist.remove(actInstance);
							worklistModel.removeRow(selection);
						}
					}
				}
			}
		});

		viewPendingButton = new JButton("View Pending Transaction");
		viewPendingButton.addActionListener((ActionEvent event) -> {
			int selection = pendingTxTable.getSelectedRow();
			if (selection != -1) {
				String txUUID = (String) pendingTxTable.getValueAt(selection, 2);
				Transaction tx = null;
				for (Transaction t : pendingTransactions.keySet()) {
					if (t.getID().toString().equals(txUUID)) {
						tx = t; 
						break;
					}
				}
				if (tx != null) {
					viewPendingTransaction(tx);
				}
			}
		});
		
		caseButton = new JButton("Launch New Case");
		caseButton.addActionListener((ActionEvent event) -> {
			int selection = petriNetTable.getSelectedRow();
			if (selection != -1) {
				String pname = (String)petriNetTable.getValueAt(selection, 0);
				if (pname != null) {
					Transaction t = new InitCaseTransaction( new InitCaseTransaction.NetNameAndCaseId(pname, UUID.randomUUID() ));
					t.setOriginator(p2pnode.whoAmI());
					p2pnode.signTransaction(t);
					if (!p2pnode.transactionService.add(t, p2pnode.whoAmI())) {
						JOptionPane.showMessageDialog(this, "Transaction could not be validated.\nThere may be a pending conflicting transaction.\nYou may retry later.", "Transaction Error (Failure)", JOptionPane.ERROR_MESSAGE);
					}
				}
			}
		});

		pnSaveButton = new JButton("Save Workflow Model");
		pnSaveButton.addActionListener((ActionEvent event) -> {
			int selection = petriNetTable.getSelectedRow();
			if (selection != -1) {
				String pname = (String)petriNetTable.getValueAt(selection, 0);
				if (pname != null) {
					JFileChooser fc = new JFileChooser();
					fc.addChoosableFileFilter(new FileNameExtensionFilter("Blockchain WFMS specification (.bws)", "bws"));
					if (fc.showSaveDialog(this) == JFileChooser.APPROVE_OPTION) {
						String fname;
						try {
							fname = fc.getSelectedFile().getCanonicalPath();
							PetriNet pn = p2pnode.workflowEngine.visibleState.getPetriNet(pname);
							if (pn != null) {
								pn.writeToFile(fname);
							}
						} catch (IOException e) {
							System.err.println("Get canonical path failed when saving workflow model");
						}
					}
				}
			}
		});
		
		pnLoadButton = new JButton("Load & Define Workflow Model");
		pnLoadButton.addActionListener((ActionEvent event) -> {
			JFileChooser fc = new JFileChooser();
			FileFilter ff = new FileNameExtensionFilter("Blockchain WFMS specification (.bws)", "bws");
			fc.addChoosableFileFilter(ff);
			if (fc.showOpenDialog(this) == JFileChooser.APPROVE_OPTION) {
				String pname = fc.getSelectedFile().getName();
				PetriNet pn = new PetriNet(pname);
				try {
					if (pn.readFromFile(fc.getSelectedFile().getCanonicalPath())) {
						Transaction t = new ModelUpdateTransaction(pn);
						t.setOriginator(p2pnode.whoAmI());
						p2pnode.signTransaction(t);
						if (!p2pnode.transactionService.add(t, p2pnode.whoAmI())) {
							JOptionPane.showMessageDialog(this, "Transaction could not be validated.\nThere may be a pending conflicting transaction.\nYou may retry later.", "Transaction Error (Failure)", JOptionPane.ERROR_MESSAGE);
						}
					}
				} catch (IOException e) {
					System.err.println("Get canonical path failed when loading workflow model");
				}
			}
		});

		worklistModel = new DefaultTableModel();
		worklistModel.addColumn("PetriNet");
		worklistModel.addColumn("CaseID");
		worklistModel.addColumn("Activity");
		worklistModel.addColumn("Resource");
		worklistTable = new JTable(worklistModel);
		worklistTable.setDefaultEditor(Object.class, null);
		worklistTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		JScrollPane wlScroll = new JScrollPane(worklistTable);//
		// wlScroll.setMinimumSize(new Dimension(500, 300));

		caseModel = new DefaultTableModel();
		caseModel.addColumn("PetriNet");
		caseModel.addColumn("CaseID");
		caseModel.addColumn("Deadlocked");
		caseModel.addColumn("Ended");
		caseModel.addColumn("Enabled Transitions");
		caseTable = new JTable(caseModel);
		caseTable.setDefaultEditor(Object.class,  null);
		caseTable.setRowSelectionAllowed(false);
		JScrollPane caseScroll = new JScrollPane(caseTable);
		
		petriNetModel = new DefaultTableModel();
		petriNetModel.addColumn("Name");
		petriNetModel.addColumn("Transitions");
		petriNetModel.addColumn("Places");
		petriNetTable = new JTable(petriNetModel);
		petriNetTable.setDefaultEditor(Object.class, null);
		petriNetTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		JScrollPane pnScroll = new JScrollPane(petriNetTable);
		
		pendingTxModel = new DefaultTableModel();
		pendingTxModel.addColumn("Origin");
		pendingTxModel.addColumn("Depth");
		pendingTxModel.addColumn("UUID");
		pendingTxModel.addColumn("Type");
		pendingTxModel.addColumn("Description");
		pendingTxTable = new JTable(pendingTxModel);
		pendingTxTable.setDefaultEditor(Object.class,  null);
		petriNetTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		JScrollPane pendingTxScroll = new JScrollPane(pendingTxTable);
		
	    Container pane = getContentPane();
        GridBagLayout gl = new GridBagLayout();
        GridBagConstraints gc = new GridBagConstraints();
        gc.anchor = GridBagConstraints.WEST;
        gc.fill = GridBagConstraints.BOTH;
        pane.setLayout(gl);
        
        gc.gridx = 0;
        gc.gridy = 0;
        gc.weightx = 1;
        gc.weighty = 0;
        pane.add(pnLoadButton, gc);
        gc.gridx = 1;
        gc.gridy = 0;
        pane.add(pnSaveButton, gc);
        gc.gridx = 0;
        gc.gridy = 1;
        gc.weighty = .5;
        gc.gridwidth = 2;
        pane.add(pnScroll, gc);
        gc.gridy = 2;
        gc.weighty = 0;
        gc.gridwidth = 2;
        pane.add(caseButton, gc);
        gc.gridy = 3;
        gc.weighty = .5;
        gc.gridwidth = 2;
        pane.add(caseScroll, gc);
        gc.gridy = 4;
        gc.weighty = 0;
        gc.gridwidth = 2;
        pane.add(workItemButton, gc);
        gc.gridy = 5;
        gc.weighty = 0.5;
        gc.gridwidth = 2;
        pane.add(wlScroll, gc);
        gc.gridy = 6;
        gc.weighty = 0;
        gc.gridwidth = 2;
        pane.add(new JLabel("Pending Workflow Actions"), gc);
        gc.gridy = 7;
        gc.weighty = 0;
        gc.gridwidth = 2;
        pane.add(viewPendingButton, gc);
        gc.gridy = 8;
        gc.weighty = 0.5;
        gc.gridwidth = 2;
        pane.add(pendingTxScroll, gc);
	}
	
	/*
	 * Pending transactions are not persisted across restarts.
	 * They are merely a hint to users
	 */
    void addToPendingTransactions(Transaction t, int depth) {
    	pendingTransactions.put(t, depth);
    	pendingTxModel.setRowCount(0);
    	for (Transaction tx : pendingTransactions.keySet()) {
    		Vector<String> newRow = new Vector<String>();
    		newRow.add(tx.getOriginator().toString());
    		newRow.add(pendingTransactions.get(tx).toString());
    		newRow.add(tx.getID().toString());
    		newRow.add(tx.getClass().getSimpleName());
    		newRow.add(tx.payload.toString());
    		pendingTxModel.addRow(newRow);
    	}
	}
    
    void removeFromPendingTransactions(Transaction t) {
    	if (pendingTransactions.remove(t) != null) {
	    	pendingTxModel.setRowCount(0);
	    	for (Transaction tx : pendingTransactions.keySet()) {
	    		Vector<String> newRow = new Vector<String>();
	    		newRow.add(tx.getOriginator().toString());
	    		newRow.add(pendingTransactions.get(tx).toString());
	    		newRow.add(tx.getID().toString());
	    		newRow.add(tx.getClass().getSimpleName());
	    		newRow.add(tx.payload.toString());
	    		pendingTxModel.addRow(newRow);
	    	}
    	}
    }
    
    protected void removeFromWorklistModel(String uuid, String tName) {
		for (int i = 0; i<worklistModel.getRowCount(); i++) {
			if (worklistModel.getValueAt(i, 1).equals(uuid) && worklistModel.getValueAt(i, 2).equals(tName)) {
				worklistModel.removeRow(i);
				break;
			}
		}
    }
    
    protected void removeCaseFromWorklistModel(UUID caseId) {
		for (int i = 0; i<worklistModel.getRowCount(); i++) {
			if (worklistModel.getValueAt(i, 0).equals(caseId.toString())) {
				worklistModel.removeRow(i);
				break;
			}
		}
    }
    
	protected void addToWorklistModel(ActivityInstance actInstance) {
		Vector<String> row = new Vector<String>();
		row.add(actInstance.getPNName());
		row.add(actInstance.getCaseId().toString());
		row.add(actInstance.getTransition().getName());
		row.add(actInstance.getTransition().getActivity().getRole());
		worklistModel.addRow(row);
	}
	
	protected void removeFromWorkflowModel(PetriNet pn) {
		String pnName = pn.getName();
		for (int i = 0; i<petriNetModel.getRowCount(); i++) {
			if (petriNetModel.getValueAt(i, 0).equals(pnName)) {
				petriNetModel.removeRow(i);
				break;
			}
		}
	}
	
	protected void addToWorkflowModel(PetriNet pn) {
		Vector<String> row = new Vector<String>();
		row.add(pn.getName());
		row.add(Integer.toString(pn.getTransitions().size()));
		row.add(Integer.toString(pn.getPlaces().size()));
		petriNetModel.addRow(row);
	}
	
	protected void removeFromCaseModel(UUID caseId) {
		for (int i = 0; i<caseModel.getRowCount(); i++) {
			if (caseModel.getValueAt(i, 1).equals(caseId.toString())) {
				caseModel.removeRow(i);
				break;
			}
		}		
	}

	protected void addToOrUpdateCaseModel(PetriNetInstance pni) {
		String caseId = pni.getId().toString();
		for (int i = 0; i<caseModel.getRowCount(); i++) {
			if (caseModel.getValueAt(i, 1).equals(caseId)) {
				caseModel.removeRow(i);
				break;
			}
		}
		Vector<String> row = new Vector<String>();
		row.add(pni.getNet().getName());
		row.add(caseId);
		row.add(pni.isDeadlocked() ? "Y" : "N");
		row.add(pni.isDone() ? "Y" : "N");
		row.add(Integer.toString(pni.getEnabled().size()));
		caseModel.addRow(row);
	}
	
	protected void resetModels() {
		worklistModel.setRowCount(0);
		petriNetModel.setRowCount(0);
		caseModel.setRowCount(0);
		pendingTxModel.setRowCount(0);
	}

	@SuppressWarnings("rawtypes")
	private TreeMap<String, Serializable> doWorkItem(ActivityInstance activityInstance) {
		JPanel panel = new JPanel();
		GridBagLayout gl = new GridBagLayout();
		GridBagConstraints gc = new GridBagConstraints();
		gc.anchor = GridBagConstraints.WEST;
		gc.fill = GridBagConstraints.BOTH;
		panel.setLayout(gl);
		
		ActivitySpec spec = activityInstance.getTransition().getActivity();
	
		int y = 0;
		for (Entry<String, Serializable> e : activityInstance.getInputValues().entrySet()) {
			gc.gridx=0;
			gc.gridy=y;
			panel.add(new Label(e.getKey()+"("+spec.getTypes().get(e.getKey())+")"), gc);
			gc.gridx=1;
			gc.gridy=y;
			TextField tf;
			if (e.getValue() != null) {
				tf = new TextField(e.getValue().toString(), BlockChainWFMSConfig.textFieldWidth);
			} else {
				System.err.println("Warning: Empty input "+e.getKey()+" in activity instance for "+activityInstance.getTransition().getName()+" in net instance ID "+activityInstance.getCaseId());
				tf = new TextField(BlockChainWFMSConfig.textFieldWidth);				
			}
			tf.setEditable(false);
			panel.add(tf, gc);
			y++;
		}
		HashMap<String, TextField> fields = new HashMap<String, TextField>();
		for (Entry<String, Serializable> e : activityInstance.getPreValues().entrySet()) {
			gc.gridx=0;
			gc.gridy=y;
			panel.add(new Label(e.getKey()+"("+spec.getTypes().get(e.getKey())+")"), gc);
			gc.gridx=1;
			gc.gridy=y;
			TextField field;
			if (e.getValue() != null) {
				field = new TextField(e.getValue().toString(), BlockChainWFMSConfig.textFieldWidth);
			} else {
				field = new TextField(BlockChainWFMSConfig.textFieldWidth);
			}
			field.setEditable(true);
			panel.add(field, gc);
			fields.put(e.getKey(), field);
			y++;
		}

		int result = JOptionPane.showConfirmDialog(this, panel, "doWorkItem: " + activityInstance.getTransition().getName() + " for case " + activityInstance.getCaseId() + " for Workflow " + activityInstance.getPNName(), JOptionPane.OK_CANCEL_OPTION);
		if (result == JOptionPane.OK_OPTION) {
			TreeMap<String, Serializable> outputs = new TreeMap<String, Serializable>();
			for (Entry<String, TextField> e : fields.entrySet()) {
				String varName = e.getKey();
				Serializable newValue = null;
				Class type = activityInstance.getTransition().getActivity().getTypes().get(varName);
				if (type == Integer.class) {
					try {
						newValue = Integer.valueOf(fields.get(varName).getText());
					} catch (NumberFormatException ex) {
						newValue = new Integer(0);
					}
				}
				if (type == String.class) {
					newValue = fields.get(varName).getText();
				}
				if (type == Long.class) {
					try {
						newValue = Long.valueOf(fields.get(varName).getText());
					} catch (NumberFormatException ex) {
						newValue = new Long(0l);
					}
				}
				if (type == Float.class) {
					try {
						newValue = Float.valueOf(fields.get(varName).getText());
					} catch (NumberFormatException ex) {
						newValue = new Float(0f);
					}
				}
				if (type == Double.class) {
					try {
						newValue = Double.valueOf(fields.get(varName).getText());
					} catch (NumberFormatException ex) {
						newValue = new Double(0d);
					}
				}
				if (type == Short.class) {
					try {
						newValue = Short.valueOf(fields.get(varName).getText());
					} catch (NumberFormatException ex) {
						newValue = new Short((short)0);
					}
				}
				if (type == Boolean.class) {
					newValue = Boolean.valueOf(fields.get(varName).getText());
				}
				if (newValue == null) {
					System.err.println("Unrecognized type or invalid conversion in doWorkItem");
				}
				outputs.put(varName, newValue);
			}
			return outputs;
		}
		return null;
	}

	private void viewPendingTransaction(Transaction tx) {		
		JPanel panel = new JPanel();
		if (tx instanceof FireTransitionTransaction) {
			panel.setLayout(new GridLayout(3, 1));
			ActivityInstance activityInstance = (ActivityInstance)tx.payload;
		
			JPanel inputPanel = new JPanel();
			inputPanel.setBorder(BorderFactory.createTitledBorder(BorderFactory.createEtchedBorder(), "Inputs"));
			GridBagLayout glInput = new GridBagLayout();
			GridBagConstraints gcInput = new GridBagConstraints();
			gcInput.anchor = GridBagConstraints.WEST;
			gcInput.fill = GridBagConstraints.BOTH;
			inputPanel.setLayout(glInput);		
			panel.add(inputPanel);
			
			JPanel outputPanel = new JPanel();		
			outputPanel.setBorder(BorderFactory.createTitledBorder(BorderFactory.createEtchedBorder(), "Outputs"));		
			GridBagLayout glOutput = new GridBagLayout();
			GridBagConstraints gcOutput = new GridBagConstraints();
			gcOutput.anchor = GridBagConstraints.WEST;
			gcOutput.fill = GridBagConstraints.BOTH;
			outputPanel.setLayout(glOutput);		
			panel.add(outputPanel);
			
			ActivitySpec spec = activityInstance.getTransition().getActivity();
		
			int y = 0;
			for (Entry<String, Serializable> e : activityInstance.getInputValues().entrySet()) {
				gcInput.gridx=0;
				gcInput.gridy=y;
				inputPanel.add(new Label(e.getKey()+"("+spec.getTypes().get(e.getKey())+")"), gcInput);
				gcInput.gridx=1;
				gcInput.gridy=y;
				TextField tf;
				if (e.getValue() != null) {
					tf = new TextField(e.getValue().toString(), BlockChainWFMSConfig.textFieldWidth);
				} else {
					System.err.println("Warning: Empty input "+e.getKey()+" in activity instance for "+activityInstance.getTransition().getName()+" in net instance ID "+activityInstance.getCaseId());
					tf = new TextField(BlockChainWFMSConfig.textFieldWidth);				
				}
				tf.setEditable(false);
				inputPanel.add(tf, gcInput);
				y++;
			}
			y = 0;
			for (Entry<String, Serializable> e : activityInstance.getPostValues().entrySet()) {
				gcOutput.gridx=0;
				gcOutput.gridy=y;
				outputPanel.add(new Label(e.getKey()+"("+spec.getTypes().get(e.getKey())+")"), gcOutput);
				gcOutput.gridx=1;
				gcOutput.gridy=y;
				TextField field;
				if (e.getValue() != null) {
					field = new TextField(e.getValue().toString(), BlockChainWFMSConfig.textFieldWidth);
				} else {
					field = new TextField(BlockChainWFMSConfig.textFieldWidth);
				}
				field.setEditable(false);
				outputPanel.add(field, gcOutput);
				y++;
			}
		} else {
			panel.setLayout(new GridLayout(1, 1));	
		}

		JPanel metaPanel = new JPanel();
		metaPanel.setBorder(BorderFactory.createTitledBorder(BorderFactory.createEtchedBorder(), "Transaction Info"));
		GridBagLayout glMeta = new GridBagLayout();
		GridBagConstraints gcMeta = new GridBagConstraints();
		gcMeta.anchor = GridBagConstraints.WEST;
		gcMeta.fill = GridBagConstraints.BOTH;
		metaPanel.setLayout(glMeta);		
		panel.add(metaPanel);

		gcMeta.gridx=0;
		gcMeta.gridy=0;
		metaPanel.add(new Label("Transaction Type"), gcMeta);
		gcMeta.gridx=1;
		metaPanel.add(new Label(tx.getClass().getSimpleName()), gcMeta);

		gcMeta.gridx=0;
		gcMeta.gridy=1;
		metaPanel.add(new Label("Originator"), gcMeta);
		gcMeta.gridx=1;
		metaPanel.add(new Label(tx.getOriginator().toString()), gcMeta);

		gcMeta.gridx=0;
		gcMeta.gridy=2;
		metaPanel.add(new Label("UUID"), gcMeta);
		gcMeta.gridx=1;
		metaPanel.add(new Label(tx.getID().toString()), gcMeta);

		gcMeta.gridx=0;
		gcMeta.gridy=3;
		metaPanel.add(new Label("Summary"), gcMeta);
		gcMeta.gridx=1;
		metaPanel.add(new Label(tx.payload.toString()), gcMeta);
		
		JOptionPane.showMessageDialog(this, panel, "viewTransaction", JOptionPane.INFORMATION_MESSAGE);
	}
}
