package ca.evermann.joerg.blockchainWFMS.chain;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Arrays;
import java.util.Base64;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

import ca.evermann.joerg.blockchainWFMS.CA.PeerCertificate;
import ca.evermann.joerg.blockchainWFMS.main.BlockChainWFMSConfig;
import ca.evermann.joerg.blockchainWFMS.p2p.P2PNode;
import ca.evermann.joerg.blockchainWFMS.p2p.messages.BlockchainSendMessage;
import ca.evermann.joerg.blockchainWFMS.p2p.messages.BlockRequestMessage;
import ca.evermann.joerg.blockchainWFMS.p2p.messages.BlockSendMessage;

public class BlockService {

	private final TransactionService transactionService;
	private final P2PNode p2pnode;

	private Set<TreeNode<Block>> blockchains = new HashSet<TreeNode<Block>>();
	private Map<String, TreeNode<Block>> knownBlocks = new HashMap<String, TreeNode<Block>>();
	private TreeNode<Block> mainBranchHead = null;

	public BlockService(P2PNode p2pnode) {
		this.p2pnode = p2pnode;
		this.transactionService = this.p2pnode.transactionService;
		this.readBlockChains();
	}

	@SuppressWarnings("unchecked")
	private synchronized void readBlockChains() {
		ObjectInputStream o = null;
		try {
			o = new ObjectInputStream(new FileInputStream(
					p2pnode.whoAmI().getHost() + "." + p2pnode.whoAmI().getPort() + ".blockchain.js"));
			blockchains = (HashSet<TreeNode<Block>>) o.readObject();
			knownBlocks = (HashMap<String, TreeNode<Block>>) o.readObject();
			mainBranchHead = (TreeNode<Block>) o.readObject();
			o.close();
			System.out.println(toString());
		} catch (IOException e) {
			System.err.println("Error while loading blockchain");
		} catch (ClassNotFoundException e) {
			System.err.println("Class not found while loading blockchain");
		}
	}

	public synchronized void persistBlockChain() {
		ObjectOutputStream o;
		try {
			o = new ObjectOutputStream(new FileOutputStream(
					p2pnode.whoAmI().getHost() + "." + p2pnode.whoAmI().getPort() + ".blockchain.js"));
			o.writeObject(blockchains);
			o.writeObject(knownBlocks);
			o.writeObject(mainBranchHead);
			o.close();
		} catch (IOException e) {
			System.err.println("Error while saving blockchain");
		}
	}

	private synchronized TreeNode<Block> getMainBranchHeadNode() {
		return mainBranchHead;
	}

	public synchronized Block getMainBranchHead() {
		if (mainBranchHead == null) return null;
		return mainBranchHead.getElement();
	}

	public int getDepth(Block b) {
		TreeNode<Block> tn = knownBlocks.get(Base64.getEncoder().encodeToString(b.getHash()));
		if (tn != null)
			return (mainBranchHead.getHeight()-tn.getHeight());
		else
			return -1;
	}

	private synchronized void append(Block block, PeerCertificate fromPeer) {
		/* 
		 * must not be known to us 
		 */
		if (!knownBlocks.containsKey(Base64.getEncoder().encodeToString(block.getHash()))) {
			/* 
			 * find the tree node that contains the previous block 
			 */
			TreeNode<Block> prevTreeNode = null;
			byte[] previousHash = block.getPreviousBlockHash();			
			if (previousHash != null) {
				prevTreeNode = knownBlocks.get(Base64.getEncoder().encodeToString(previousHash));
			}
			/*
			 * We don't have a tree node to append this to
			 */
			if (prevTreeNode == null) {
				/*
				 * A Genesis block (pre hash is null) and we do not already 
				 * have a chain with a genesis block so let's start a new root
				 * 
				 * Note: We will not accept a second genesis block
				 */
				if (previousHash == null) {
					if (mainBranchHead != null) {
						System.err.println("Already have a chain, aborting append for block with hash "+block.toString());
						return;
					}
					System.err.println("genesis block");
					/*
					 * start a new blockchain root, remove the transactions from pool, 
					 * set the main branch head, and send to peers
					 */
					if (validateBlock(block)) {
						mainBranchHead = new TreeNode<Block>(prevTreeNode, block);
						blockchains.add(mainBranchHead);
						knownBlocks.put(Base64.getEncoder().encodeToString(block.getHash()), mainBranchHead);
						transactionService.remove(block.getTransactions());
						sendBlockExceptTo(block, fromPeer);
						/*
						 * Notify workflow engine of change, so it can requery the blockchain for most current state
						 */
						p2pnode.workflowEngine.updateHead(mainBranchHead.getElement());
					} else {
						System.err.println("ERROR: Cannot verify transactions in block \n"+block.toString()+"\nCancelling block append");
						return;
					}
				}
				/* 
				 * Orphan block (but not a genesis block)
				 * It refers to a previous block that we don't have 
				 */
				if (previousHash != null) {
					System.err.println("orphan block");
					/*
					 * Add it to our chains and request the previous block from the peer we
					 * got this from
					 */
					TreeNode<Block> newTreeNode = new TreeNode<Block>(prevTreeNode, block);
					blockchains.add(newTreeNode);
					knownBlocks.put(Base64.getEncoder().encodeToString(block.getHash()), newTreeNode);
					p2pnode.sendMessageTo(fromPeer, new BlockRequestMessage(previousHash));
				}
			} else {
				/*
				 * Appending to main branch head
				 */
				if (prevTreeNode == mainBranchHead) {
					System.err.println("main branch");
					/*
					 * appending to main branch, remove transactions in block from pool,
					 * send block to peers
					 */
					if (validateBlock(block)) {
						mainBranchHead = new TreeNode<Block>(prevTreeNode, block);
						knownBlocks.put(Base64.getEncoder().encodeToString(block.getHash()), mainBranchHead);
						transactionService.remove(block.getTransactions());
						sendBlockExceptTo(block, fromPeer);
						/*
						 * Notify workflow engine of change, so it can requery the blockchain for most current state
						 */
						p2pnode.workflowEngine.updateHead(mainBranchHead.getElement());
					} else {
						System.err.println("ERROR: Cannot verify transactions in block \n"+block.toString()+"\nCancelling block append");
						return;
					}
				} else {
					/*
					 * Appending to side branch head
					 */
					System.err.println("side branch");
					if (prevTreeNode.getHeight() < getMainBranchHeadNode().getHeight()) {
						/* 
						 * Side branch is still lower than main branch, nothing to do, just add it
						 */
						knownBlocks.put(Base64.getEncoder().encodeToString(block.getHash()), new TreeNode<Block>(prevTreeNode, block));
					}
					if (prevTreeNode.getHeight() == getMainBranchHeadNode().getHeight()) {
						/*
						 * Side branch is the same height as main branch; with this new block, it will be higher
						 * and become the new main branch
						 */
						System.err.println("** New main branch, reorganizing **");
						/*
						 * find the common ancestor of the previous tree node
						 * and the current main branch
						 */
						TreeNode<Block> commonAncestor = prevTreeNode.commonAncestor(getMainBranchHeadNode());
						/* 
						 * The main chain goes until here now, we note the old main head
						 */
						TreeNode<Block> oldMainBranchHead = mainBranchHead;
						mainBranchHead = commonAncestor;
						p2pnode.workflowEngine.resetHead(mainBranchHead.getElement());
						/*
						 * We then add each block in the former side branch 
						 * on top of the new mainBranchHead
						 * 
						 * Because we traverse the chain backwards, we store blocks in a
						 * check list so that we can check their transactions in the right order
						 */
						Vector<TreeNode<Block>> checkList = new Vector<TreeNode<Block>>();
						for (TreeNode<Block> tn = prevTreeNode; tn != commonAncestor && tn != null; tn = tn.getBough()) {
							checkList.insertElementAt(tn, 0);
						}
						/*
						 * Now we verify the check list blocks in the right order
						 */
						for (TreeNode<Block> checkNode : checkList) {
							if (validateBlock(checkNode.getElement())) {
								mainBranchHead = checkNode;
								p2pnode.workflowEngine.resetHead(mainBranchHead.getElement());
							} else {
								System.err.println("ERROR: Cannot verify transactions in block \n"+checkNode.getElement().toString()+"\nAborting chain reorg");
								mainBranchHead = oldMainBranchHead;
								p2pnode.workflowEngine.resetHead(mainBranchHead.getElement());
								return;
							}							
						}
						/*
						 * Also make sure the new block that started this reorg is added here to be checked
						 */
						if (!validateBlock(block)) {
							System.err.println("ERROR: Cannot validate transactions in new block \n"+block.toString()+"\nAborting chain reorg");
							mainBranchHead = oldMainBranchHead;
							p2pnode.workflowEngine.resetHead(mainBranchHead.getElement());
							return;
						}
						/*
						 * The transactions in the old main branch descendants get added back to the 
						 * transaction pool
						 */
						for (TreeNode<Block> tn = oldMainBranchHead; tn != commonAncestor && tn != null; tn=tn.getBough()) {
							transactionService.bulkAddFromMultipleBlocks(tn.getElement().getTransactions());
						}
						/*
						 * the transactions from the new main branch head back to the common ancestor
						 * can now be removed from the pool
						 */
						for (TreeNode<Block> tn = mainBranchHead; tn != commonAncestor && tn != null; tn = tn.getBough()) {
							transactionService.remove(tn.getElement().getTransactions());
						}
						/*
						 * When we are here, we can create a new TreeNode
						 * and send the block to our peers
						 */
						mainBranchHead = new TreeNode<Block>(prevTreeNode, block);
						knownBlocks.put(Base64.getEncoder().encodeToString(block.getHash()), mainBranchHead);
						transactionService.remove(block.getTransactions());
						sendBlockExceptTo(block, fromPeer);
						p2pnode.workflowEngine.resetHead(mainBranchHead.getElement());
					}
				}
			}

			/* Do we have an orphan block that refers to this block? */
			Set<TreeNode<Block>> orphans = new HashSet<TreeNode<Block>>();
			for (TreeNode<Block> o : blockchains) {
				if (o.getElement().getPreviousBlockHash() == block.getHash()) {
					orphans.add(o);
				}
			}
			/*
			 * For each orphan that depends on this block, remove it from the orphan set and
			 * then append it as a normal append operation
			 */
			for (TreeNode<Block> o : orphans) {
				blockchains.remove(o);
				append(o.getElement(), fromPeer);
			}
			/*
			 * All done, report final result
			 */
			System.out.println(toString());
			System.out.println(transactionService.toString());
		}
	}

	public synchronized void receiveBlockchain(List<Block> blocks, PeerCertificate fromNode) {
		/*
		 * We stop the miner so that when we do a re-organization of the chain on
		 * appending a block, the temporary transactions that we may insert into the
		 * memory pool don't get picked up for mining
		 */
		boolean miningServicePaused = p2pnode.miningService.paused;
		p2pnode.miningService.paused = true;
		p2pnode.workflowEngine.disableWorkflowUI();
		for (Block b : blocks) {
			if (verify(b)) {
				append(b, fromNode);
			}
		}
		persistBlockChain();
		transactionService.checkOrphans();
		p2pnode.workflowEngine.enableWorkflowUI();
		p2pnode.miningService.paused = miningServicePaused;
	}

	public synchronized void receiveBlock(Block b, PeerCertificate fromNode) {
		if (verify(b)) {
			/*
			 * We stop the miner so that when we do a re-organization of the chain on
			 * appending a block, the temporary transactions that we may insert into the
			 * memory pool don't get picked up for mining
			 */
			boolean miningServicePaused = p2pnode.miningService.paused;
			p2pnode.miningService.paused = true;
			p2pnode.workflowEngine.disableWorkflowUI();
			append(b, fromNode);
			persistBlockChain();
			p2pnode.workflowEngine.enableWorkflowUI();
			p2pnode.miningService.paused = miningServicePaused;
		}
	}

	public synchronized void sendBlockchainTo(PeerCertificate recipient, Vector<byte[]> fromHashes) {
		TreeNode<Block> commonAnc = null;
		for (byte[] fromHash : fromHashes) {
			commonAnc = knownBlocks.get(Base64.getEncoder().encodeToString(fromHash));
			if (commonAnc != null) {
				break;
			}
		}
		/*
		 * commonAnc now contains the latest common block, or null
		 * 
		 * now we assemble a list of blocks from the chain head in reverse order until
		 * the latest common block
		 * 
		 */
		Vector<Block> blockList = new Vector<Block>();
		TreeNode<Block> tn = mainBranchHead;
		while (tn != null && tn != commonAnc) {
			blockList.insertElementAt(tn.getElement(), 0);
			tn = tn.getBough();
		}
		p2pnode.sendMessageTo(recipient, new BlockchainSendMessage(blockList));
	}

	private void sendBlockExceptTo(Block b, PeerCertificate except) {
		p2pnode.sendMessageExceptTo(new BlockSendMessage(b), except);
	}

	public void sendBlockTo(PeerCertificate recipient, byte[] blockHash) {
		Block b = knownBlocks.get(Base64.getEncoder().encodeToString(blockHash)).getElement();
		if (b != null) {
			p2pnode.sendMessageTo(recipient, new BlockSendMessage(b));
		}
	}

	private synchronized boolean isTransactionInNodeOrTrunk_(TreeNode<Block> n, Transaction t) {
		if (n.getElement().getTransactions().contains(t)) {
			return true;
		} else {
			if (n.getBough() != null) {
				return isTransactionInNodeOrTrunk_(n.getBough(), t);
			} else {
				return false;
			}
		}
	}

	synchronized boolean isTransactionInMainChain(Transaction t) {
		if (mainBranchHead == null)
			return false;
		else
			return isTransactionInNodeOrTrunk_(mainBranchHead, t);
	}

	private synchronized void _getChainHashesForPeerRequest(int lvl, TreeNode<Block> tn, Vector<byte[]> hashes) {
		for (int i = 0; i < Math.round(Math.exp(lvl)); i++) {
			if (tn.getBough() == null) {
				// We are at the root
				if (i > 0) {
					// We were not given the root as input
					hashes.add(tn.getElement().getHash());
					System.out.println("Have Hash: " + Base64.getEncoder().encodeToString(hashes.lastElement()));
				}
				return;
			}
			tn = tn.getBough();
		}
		hashes.add(tn.getElement().getHash());
		System.out.println("Have Hash: " + Base64.getEncoder().encodeToString(hashes.lastElement()));
		_getChainHashesForPeerRequest(++lvl, tn, hashes);
	}

	public synchronized Vector<byte[]> getChainHashesForPeerRequest() {
		Vector<byte[]> hashes = new Vector<byte[]>();
		if (mainBranchHead == null) {
			// We have no chain
			return hashes;
		} else {
			hashes.add(mainBranchHead.getElement().getHash());
			System.out.println("Have Hash: " + Base64.getEncoder().encodeToString(hashes.lastElement()));
			_getChainHashesForPeerRequest(0, mainBranchHead, hashes);
			return hashes;
		}
	}

	private boolean verify(Block block) {
		// correct hashes
		if (!Arrays.equals(block.getHash(), block.calculateHash())) {
			System.err.println("  Block hash fail");
			return false;
		}
		if (!Arrays.equals(block.getMerkleRoot(), block.calculateMerkleRoot())) {
			System.err.println("  Merkle root fail");
			return false;
		}
		// too new
		if ((block.getTimestamp() - System.currentTimeMillis()) > BlockChainWFMSConfig.maxBlockNewness) {
			System.err.println("  Block too new");
			return false;
		}
		// too old
		if (getMainBranchHeadNode() != null) {
			if ((getMainBranchHeadNode().getElement().getTimestamp()
					- block.getTimestamp()) > BlockChainWFMSConfig.maxBlockAge) {
				System.err.println("  Block too old");
				return false;
			}
		}
		// no transactions
		if (block.getTransactions().size() == 0) {
			System.err.println("  Block has no transactions");
			return false;
		}
		// too many transactions
		if (block.getTransactions().size() > BlockChainWFMSConfig.maxTransactionsPerBlock) {
			System.err.println("  Block has too many transactions");
			return false;
		}
		// too easy
		if (block.getLeadingZerosCount() < BlockChainWFMSConfig.difficulty) {
			System.err.println("  Block is too easy");
			return false;
		}
		return true;
	}
	
	private synchronized boolean validateBlock(Block block) {
		for (Transaction t : block.getTransactions()) {
			/*
			 * We call the workflow engine to validate each transaction
			 * We do not consider pending transactions (=null), only the
			 * chain head
			 */
			if (!p2pnode.workflowEngine.validateTransaction(t, null)) {
				return false;
			}
		}
		return true;
	}

	@Override
	public String toString() {
		String s = new String("Blockchains:\n");
		for (TreeNode<Block> n : blockchains) {
			s = s.concat(n.toString());
			s = s.concat("\n");
		}
		if (mainBranchHead != null) {
			s = s.concat("\nMain branch head: ")
					.concat(Base64.getEncoder().encodeToString(mainBranchHead.getElement().getHash()));
		}
		return s;
	}

	public Block getPredecessor(Block b) {
		if (b == null) return null;
		if (b.getPreviousBlockHash() == null) return null;
		return knownBlocks.get(Base64.getEncoder().encodeToString(b.getPreviousBlockHash())).getElement();
	}
}