package ca.evermann.joerg.blockchainWFMS.p2p;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.util.*;
import java.util.concurrent.*;

import ca.evermann.joerg.blockchainWFMS.CA.PeerCertificate;
import ca.evermann.joerg.blockchainWFMS.chain.Block;
import ca.evermann.joerg.blockchainWFMS.chain.BlockService;
import ca.evermann.joerg.blockchainWFMS.chain.MiningService;
import ca.evermann.joerg.blockchainWFMS.chain.Transaction;
import ca.evermann.joerg.blockchainWFMS.chain.TransactionService;
import ca.evermann.joerg.blockchainWFMS.p2p.messages.BlockSendMessage;
import ca.evermann.joerg.blockchainWFMS.p2p.messages.BlockchainRequestMessage;
import ca.evermann.joerg.blockchainWFMS.p2p.messages.BlockchainSendMessage;
import ca.evermann.joerg.blockchainWFMS.p2p.messages.Message;
import ca.evermann.joerg.blockchainWFMS.p2p.messages.PeersRequestMessage;
import ca.evermann.joerg.blockchainWFMS.p2p.messages.PeersSendMessage;
import ca.evermann.joerg.blockchainWFMS.p2p.messages.TransactionPoolRequestMessage;
import ca.evermann.joerg.blockchainWFMS.p2p.messages.TransactionPoolSendMessage;
import ca.evermann.joerg.blockchainWFMS.p2p.messages.TransactionSendMessage;
import ca.evermann.joerg.blockchainWFMS.workflow.WorkflowEngine;

public class P2PNode {

	private InboundServer inServer;
	private OutboundServer outServer;
	private InboundMsgHandler msgHandler;
	private Thread inServerThread;
	private Thread outServerThread;
	private Thread msgHandlerThread;
	private Thread miningThread;

	Set<PeerCertificate> knownPeers;
	private Set<String> additionalHosts;
	HashMap<String, PeerCertificate> knownPeersByHostPort;
	HashMap<PublicKey, PeerCertificate> knownPeersByPublicKey;
	List<BlockingQueue<Message>> inboundQueues;
	Map<PeerCertificate, BlockingQueue<Message>> outboundQueues;
	HashMap<String, PeerConnection> activeConnections;
	
	PeerCertificate myCertificate;
	PrivateKey mySigningKey;

	public TransactionService transactionService;
	public BlockService blockService;
	public MiningService miningService;
	public WorkflowEngine workflowEngine;
	
	public boolean netIsUp = false;
	
/* 
 * Startup and Shutdown 
 * 
 * */
	
	public P2PNode(Set<PeerCertificate> knownPeers, PeerCertificate myCertificate, PrivateKey mySigningKey, Set<String> additionalHosts) {

		this.knownPeers = knownPeers;
		this.additionalHosts = additionalHosts;
		this.knownPeersByHostPort = new HashMap<String, PeerCertificate>();
		this.knownPeersByPublicKey = new HashMap<PublicKey, PeerCertificate>();
		for (PeerCertificate pc : knownPeers) {
			this.knownPeersByHostPort.put(pc.getHost()+pc.getPort(), pc);
			this.knownPeersByPublicKey.put(pc.getKey(), pc);
		}
		this.myCertificate = myCertificate;
		this.mySigningKey = mySigningKey;
		
		this.transactionService = new TransactionService(this);
		this.blockService = new BlockService(this);
		this.workflowEngine = new WorkflowEngine(this);
		this.miningService = new MiningService(this);
		
		netUp();
		
		miningThread = new Thread(miningService, "Mining Service Thread");
		miningThread.start();
		this.workflowEngine.enableWorkflowUI();
	}
	
	public void Shutdown() {
		savePeers();
		this.blockService.persistBlockChain();
		this.transactionService.persistTransactionPool();
		
		miningService.stop = true;
		try {
			miningThread.join();
		} catch (InterruptedException e) {
			System.err.println("Interrupted while joinin miningThread");
		}
		
		netDown();
	}

	public void netUp() {
		if (netIsUp) return;
		
		this.inboundQueues = new CopyOnWriteArrayList<BlockingQueue<Message>>();
		this.outboundQueues = new HashMap<PeerCertificate, BlockingQueue<Message>>();
		this.activeConnections = new HashMap<String, PeerConnection>();
		
		try {
			outServer = new OutboundServer(this, additionalHosts);
		} catch (IOException e1) {
			e1.printStackTrace();
		}
		outServerThread = new Thread(outServer, "Outbound Server Thread");
		outServerThread.start();
		
		try {
			inServer = new InboundServer(this);
		} catch (IOException e) {
			e.printStackTrace();
		}
		inServerThread = new Thread(inServer, "Inbound Server Thread");
		inServerThread.start();
		
		msgHandler = new InboundMsgHandler(this);
		msgHandlerThread = new Thread(msgHandler, "Inbound Message Handler Thread");
		msgHandlerThread.start();
		
		/* after we've started all the services
		 * request the latest updates to transaction pool
		 * and blockchain from our peers
		 */
		try {
			Thread.sleep(2000);
		} catch (InterruptedException e) {
			System.err.println("p2p node woken from sleep");
		}
		// Get peer information
		sendMessage(new PeersRequestMessage());
		// Get blockchain information
		sendMessage(new BlockchainRequestMessage(this.blockService.getChainHashesForPeerRequest()));
		// Get transaction pool information
		sendMessage(new TransactionPoolRequestMessage());

		netIsUp = true;
	}
	
	public void netDown() {
		if (!netIsUp) return;
		
		savePeers();
		
		inServer.stop = true;
		outServer.stop = true;
		msgHandler.stop = true;

		try {
			inServer.serverSocket.close();
		} catch (IOException e1) {
			System.err.println("Error while closing server socket");
		}
		try {
			inServerThread.join();
		} catch (InterruptedException e) {
			System.err.println("Interrupted while joining inServerThread");
		}
		try {
			outServerThread.join();
		} catch (InterruptedException e) {
			System.err.println("Interrupted while joining outServerThread");
		}
		try {
			msgHandlerThread.join();
		} catch (InterruptedException e) {
			System.err.println("Interrupted while joining msgHandlerThread");
		}
		
		netIsUp = false;
	}
/* 
 * Basic Messaging 
 * 
 * */
	
	protected List<BlockingQueue<Message>> getInboundQueues() {
		return this.inboundQueues;
	}
	
	public void sendMessage(Message msg) {
		msg.setSender(myCertificate);
		msg.signContent(mySigningKey);
		for (BlockingQueue<Message> q : outboundQueues.values()) {
			try {
				q.put(msg);
			} catch (InterruptedException e) {
				System.err.println("Interrupted while broadcasting");
				continue;
			}
		}
	}
	
	public void sendMessageExceptTo(Message msg, PeerCertificate except) {
		msg.setSender(myCertificate);
		msg.signContent(mySigningKey);
		for (Map.Entry<PeerCertificate, BlockingQueue<Message>> me : outboundQueues.entrySet()) {
			if (!me.getKey().equals(except)) {
				try {
					me.getValue().put(msg);
				} catch (InterruptedException e) {
					System.err.println("Interrupted while broadcasting");
					continue;
				}
			}
		}		
	}
	
	public void sendMessageTo(PeerCertificate recipient, Message msg) {
		msg.setSender(myCertificate);
		msg.signContent(mySigningKey);
		try {
			BlockingQueue<Message> q = outboundQueues.get(recipient);
			if (q != null)
				q.put(msg);
		} catch (InterruptedException e) {
			System.err.println("Interrupted while message sending");
		}
	}
	
/* 
 * Peer Management 
 * 
 * */

	public void addPeers(List<PeerCertificate> hosts) {
		for (PeerCertificate h : hosts) {
			if (!h.equals(myCertificate)) {
				if (this.knownPeers.add(h)) {
					this.knownPeersByHostPort.put(h.getHost()+h.getPort(), h);
					this.knownPeersByPublicKey.put(h.getKey(), h);
					synchronized(this.outServer.availableHosts) {
						this.outServer.availableHosts.add(h.getHost()+":"+h.getPort());
					}
				}
			}
		}
	}
	
	public void sendKnownPeersTo(PeerCertificate recipient) {
    	ArrayList<PeerCertificate> newHosts = new ArrayList<PeerCertificate>();
    	for (PeerCertificate h : this.knownPeers) {
    		newHosts.add(h);
    	}
		sendMessageTo(recipient, new PeersSendMessage(newHosts));
	}
	
	private void savePeers() {
		ObjectOutputStream o;
		try {
			o = new ObjectOutputStream(new FileOutputStream(myCertificate.getHost()+"."+myCertificate.getPort()+".knownPeers.js"));
			o.writeObject(this.knownPeers);
			o.close();
		} catch (IOException e) {
			System.err.println("Error while saving known peers");
		}
	}
	
	public PeerCertificate getCertificate(String host, Integer port) {
		return this.knownPeersByHostPort.get(host+port);		
	}
	
	public PeerCertificate getCertificate(PublicKey key) {
		return this.knownPeersByPublicKey.get(key);
	}

	public PeerCertificate whoAmI() {
		return this.myCertificate;
	}
	
/* 
 * Message Handlers 
 * 
 * */
	
	public void sendBlockchainTo(PeerCertificate recipient, Vector<byte[]> fromHashes) {
		blockService.sendBlockchainTo(recipient, fromHashes);
	}

	@SuppressWarnings("unchecked")
	public void receiveBlockChain(BlockchainSendMessage msg) {
		blockService.receiveBlockchain( (List<Block>)msg.payload, msg.getSender() );
	}
	
	public void receiveBlock(BlockSendMessage msg) {
		blockService.receiveBlock( (Block)msg.payload, msg.getSender() );
	}
	
	public void sendTransactionPoolTo(PeerCertificate recipient) {
		transactionService.sendTransactionPoolTo(recipient);
	}

	public void sendBlockTo(PeerCertificate recipient, byte[] blockHash) {
		blockService.sendBlockTo(recipient,  blockHash);
	}
	
	@SuppressWarnings("unchecked")
	public void receiveTransactionPool(TransactionPoolSendMessage msg) {
		ArrayList<Transaction> txPool = (ArrayList<Transaction>)msg.payload;
		for (Transaction t : txPool) {
			transactionService.add(t, msg.getSender());
		}
	}
	
	public void receiveTransaction(TransactionSendMessage msg) {
		transactionService.add( (Transaction)msg.payload, msg.getSender() );
	}

/*
 * Transaction Signing Service
 * 
 * */
	public void signTransaction(Transaction t) {
		t.signContent(mySigningKey);
	}

}
