package ca.evermann.joerg.blockchainWFMS.workflow;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import ca.evermann.joerg.blockchainWFMS.PetriNet.PetriNet;
import ca.evermann.joerg.blockchainWFMS.chain.Block;
import ca.evermann.joerg.blockchainWFMS.chain.Transaction;
import ca.evermann.joerg.blockchainWFMS.main.BlockChainWFMSConfig;
import ca.evermann.joerg.blockchainWFMS.p2p.P2PNode;

public class WorkflowEngine {

	private WorklistUI					worklistUI;
	private P2PNode						p2pnode;
	
	private Map<UUID, PetriNetInstance> runningInstances;
	private Map<String, PetriNet>		knownPetriNets;
	
	public WorkflowEngine(P2PNode p2pnode) {
		this.p2pnode = p2pnode;
		this.worklistUI = new WorklistUI(p2pnode);
		this.runningInstances = new HashMap<UUID, PetriNetInstance>();
		this.knownPetriNets = new HashMap<String, PetriNet>();
		readBlockChain();
	}

	public void enableWorkflowUI() {
		this.worklistUI.setEnabled(true);
		this.worklistUI.setVisible(true); 
	}
	
	public void disableWorkflowUI() {
		this.worklistUI.setEnabled(false);
	}

	/*
	 * This re-reads the entire chain up to the given head node
	 */
	private void readBlockChainTo(Block head) {
		knownPetriNets.clear();
		runningInstances.clear();

		worklistUI.resetModels();
		/*
		 * Must traverse chain twice: We must have the PetriNet definitions before we
		 * can meaningfully read the instance states, as they refer to the Petri nets
		 */
		Block b = head;
		while (b != null) {
			for (Transaction t : b.getTransactions()) {
				if (t instanceof ca.evermann.joerg.blockchainWFMS.workflow.transactions.ModelDefinitionTransaction) {
					PetriNet pn = (PetriNet)t.payload;
					// We only get the most recent
					if (!knownPetriNets.containsKey(pn.getName())) {
						knownPetriNets.put(pn.getName(), pn);
						if (p2pnode.blockService.getDepth(b)  >= BlockChainWFMSConfig.minConfirmDepth) {
							worklistUI.updatePetriNetList(pn);
						} else {
							worklistUI.addToPendingTransactions(t);
						}
					}
				}
			}
			b = p2pnode.blockService.getPredecessor(b);
		}
		b = head;
		while (b != null) {
			for (Transaction t : b.getTransactions()) {
				if (t instanceof ca.evermann.joerg.blockchainWFMS.workflow.transactions.InstanceStateTransaction) {
					PetriNetInstance pni = (PetriNetInstance)t.payload;
					// We only get the most recent
					if (!runningInstances.containsKey(pni.getId())) {
						runningInstances.put(pni.getId(), pni);
						if (p2pnode.blockService.getDepth(b)  >= BlockChainWFMSConfig.minConfirmDepth) {
							worklistUI.updateWorklist(pni);
						} else {
							worklistUI.addToPendingTransactions(t);
						}
					}
				}
			}
			b = p2pnode.blockService.getPredecessor(b);
		}
	}
	
	/*
	 * This re-reads the entire chain up to the current main branch head
	 */
	private void readBlockChain() {
		readBlockChainTo(p2pnode.blockService.getMainBranchHead());
	}

	/*
	 * This is called by the block service when a block is added to the main branch
	 */
	public void updateHead(Block b) {
		for (Transaction t : b.getTransactions()) {
			if (t instanceof ca.evermann.joerg.blockchainWFMS.workflow.transactions.ModelDefinitionTransaction) {
				PetriNet pn = (PetriNet)t.payload;
				knownPetriNets.put(pn.getName(), pn);
				worklistUI.addToPendingTransactions(t);
			}
			if (t instanceof ca.evermann.joerg.blockchainWFMS.workflow.transactions.InstanceStateTransaction) {
				PetriNetInstance pni = (PetriNetInstance)t.payload;
				runningInstances.put(pni.getId(), pni);
				worklistUI.addToPendingTransactions(t);
			}
		}
		// get down into the chain from the head block until 
		// we are at the depth that is considered confirmed
		while (b != null && !(p2pnode.blockService.getDepth(b)  >= BlockChainWFMSConfig.minConfirmDepth)) {
			b = p2pnode.blockService.getPredecessor(b);
		}
		if (b != null) {
			// newly considered confirmed block at the right depth
			for (Transaction t : b.getTransactions()) {
				if (t instanceof ca.evermann.joerg.blockchainWFMS.workflow.transactions.ModelDefinitionTransaction) {
					PetriNet pn = (PetriNet)t.payload;
					worklistUI.updatePetriNetList(pn);
				}
				if (t instanceof ca.evermann.joerg.blockchainWFMS.workflow.transactions.InstanceStateTransaction) {
					PetriNetInstance pni = (PetriNetInstance)t.payload;
					worklistUI.updateWorklist(pni);
				}
				// remove from the local pending transactions, if it exists in there
				worklistUI.removeFromPendingTransactions(t);
			}
		}
	}
	
	/*
	 * This is called when the block service resets the branch head
	 */
	public void resetHead(Block b) {
		readBlockChainTo(b);
	}
	
	private PetriNetInstance getPetriNetInstance(PetriNet pn, UUID caseId) {
		PetriNetInstance pni = runningInstances.get(caseId);
		if (pni == null) return null;
		if (!pni.getNet().equals(pn)) {
			System.err.println("Wrong PN for caseId "+caseId.toString()+". Expected+"+pn.getName()+", got "+pni.getNet().getName());
			return null;
		}
		return pni;
	}
	
	protected PetriNet getPetriNet(String pnName) {
		return knownPetriNets.get(pnName);
	}
	
	public boolean validateTransaction(Transaction t, Set<Transaction> pendingTx) {
		if (t != null) {
			if (t.payload == null) 
				return false;
			switch (t.getClass().getName()) {
			case "ca.evermann.joerg.blockchainWFMS.workflow.transactions.InstanceStateTransaction":
				PetriNetInstance newPNI = (PetriNetInstance)t.payload;
				// petri net must be known to us
				PetriNet pn = this.knownPetriNets.get(newPNI.getNet().getName());
				if (pn == null) {
					return false;
				}
				// if we don't have a current state, this must be the initial state
				PetriNetInstance currentPNI = getPetriNetInstance(newPNI.getNet(), newPNI.getId());
				// if we don't have a current state, this must be the initial state
				if (currentPNI == null && newPNI.isInitial()) {
					return newPNI.checkConstraints();
				}
				if (currentPNI == null) {
					return false;
				}
				if (!newPNI.checkConstraints()) {
					return false;
				}
				if (!newPNI.isReachableFrom(currentPNI)) {
					return false;
				}
				// if it's not the initial state, the new state must be reachable from
				// our current state, and all pending states!
				boolean valid = true;
				if (pendingTx != null) {
					for (Transaction pending : pendingTx) {
						if (pending.getClass().getName().equals("ca.evermann.joerg.blockchainWFMS.workflow.transactions.InstanceStateTransaction")) {
							PetriNetInstance pendingPNI = (PetriNetInstance) pending.payload;
							if (pendingPNI.getId().equals(newPNI.getId())) {
								valid = valid && newPNI.isReachableFrom(pendingPNI);
							}
						}
					}
				}
				return valid;
			case "ca.evermann.joerg.blockchainWFMS.workflow.transactions.ModelDefinitionTransaction":
				PetriNet	petriNet = (PetriNet) t.payload;
				if (this.knownPetriNets.containsKey(petriNet.getName())) {
					return false;
				}
				return true;
			default:
				return true;
			}
		}
		return false;
	}
	
	@Override
	public String toString() {
		String s = new String("Running Instance:\n");
		for (UUID u : runningInstances.keySet()) {
			s = s.concat(" " + u.toString() + "\n");
		}
		s = s.concat("Known Petri nets:\n");
		for (String st : knownPetriNets.keySet()) {
			s = s.concat(" " + st + "\n");			
		}
		return s;
	}

}
