import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.Provider;
import java.security.Provider.Service;
import java.security.SecureRandom;
import java.security.Security;
import java.security.Signature;
import java.security.SignatureException;
import java.security.interfaces.DSAPrivateKey;
import java.security.interfaces.DSAPublicKey;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.ECPublicKey;
import java.security.spec.ECGenParameterSpec;

import org.apache.commons.codec.binary.Hex;

public class DigitalSignatureTest {

	// This holds our random number generator used for both demos
	private static SecureRandom secRnd;

	// This is the message we're going to sign
	private static final String message = "Hello Blockweek 2019";
	/* 
	 * This method demonstrates signing and signature verification with
	 * Elliptic Curve Cryptography
	 */
	public static void ellipticCurvesDemo()  throws NoSuchAlgorithmException, InvalidAlgorithmParameterException, InvalidKeyException, SignatureException {
		
		// Select an Elliptic Curve (discrete logarithm) based key generator
		KeyPairGenerator keyPairGen = KeyPairGenerator.getInstance("EC");
		
		// Print the supported curves by various providers
		System.out.println("Available Curves:");
		for (Provider prov : Security.getProviders("AlgorithmParameters.EC") ) {
			System.out.println("  " + prov + " " + prov.getService("AlgorithmParameters", "EC").getAttribute("SupportedCurves")) ;
		}
		
		// Set algorithm parameters for key pair generator
		ECGenParameterSpec curveSpec = new ECGenParameterSpec("secp256k1");

		// Initialize the key pair generator with the curve specification and the random number generator
	    keyPairGen.initialize(curveSpec, secRnd);	
	    
	    // Generate a key pair
		KeyPair ECKeyPair = keyPairGen.generateKeyPair();
		ECPublicKey publicKey = (ECPublicKey) ECKeyPair.getPublic();
		ECPrivateKey privateKey = (ECPrivateKey) ECKeyPair.getPrivate();

		System.out.println("Private Key: ");
		System.out.println("  " + privateKey.getAlgorithm());
		System.out.println("  " + privateKey.getFormat());
		System.out.println("  S=" + privateKey.getS());
		System.out.println("  " + privateKey.getParams());
		System.out.println("  " + Hex.encodeHexString(privateKey.getEncoded()));
		System.out.println("Public Key: ");
		System.out.println("  " + publicKey.getAlgorithm());
		System.out.println("  " + publicKey.getFormat());
		System.out.println("  W.x=" + publicKey.getW().getAffineX());
		System.out.println("  W.y=" + publicKey.getW().getAffineY());
		System.out.println("  " + publicKey.getParams());
		System.out.println("  " + Hex.encodeHexString(publicKey.getEncoded()));

		// Print list of supported digital signature algorithms
		System.out.println("Digital signature algorithms:");
		for (String algorithm : Security.getAlgorithms("Signature")) {
			System.out.println("  "+algorithm);
		}
		
		// Select a digital signature algorithm
		Signature ECDSA = Signature.getInstance("SHA1withECDSA");

		// Initialize it with the private key and the random number generator
		ECDSA.initSign(privateKey, secRnd);

		// Set the message bytes into the signature algorithm
		ECDSA.update(message.getBytes());
		
		// And sign it, getting the signature
		byte[] signature = ECDSA.sign();
		
		System.out.println("Message Signature: ");
		System.out.println(Hex.encodeHex(signature));

		// Initialize the ECDSA algorithm for verification
		ECDSA.initVerify(publicKey);
		
		// Set the message bytes into the signature algorithm
		ECDSA.update(message.getBytes());
		
		// And verify it
		boolean isSignatureVerified = ECDSA.verify(signature);		
		System.out.println("Signature is verified: " + isSignatureVerified);
	}
	
	/* 
	 * This method demonstrates signing and signature verification with
	 * Elliptic Curve Cryptography
	 */
	public static void primeFactorizationDemo()  throws NoSuchAlgorithmException, InvalidAlgorithmParameterException, InvalidKeyException, SignatureException {
		
		// Select an Elliptic Curve (discrete logarithm) based key generator
		KeyPairGenerator keyPairGen = KeyPairGenerator.getInstance("DSA");
		
		// Initialize the key pair generator with the parameters and the random number generator
	    keyPairGen.initialize(512, secRnd);	
	    
	    // Generate a key pair
		KeyPair DSAKeyPair = keyPairGen.generateKeyPair();
		DSAPublicKey publicKey = (DSAPublicKey) DSAKeyPair.getPublic();
		DSAPrivateKey privateKey = (DSAPrivateKey) DSAKeyPair.getPrivate();

		System.out.println("Private Key: ");
		System.out.println("  " + privateKey.getAlgorithm());
		System.out.println("  " + privateKey.getFormat());
		System.out.println("  x=" + privateKey.getX());
		System.out.println("  " + privateKey.getParams());
		System.out.println("  " + Hex.encodeHexString(privateKey.getEncoded()));
		System.out.println("Public Key: ");
		System.out.println("  " + publicKey.getAlgorithm());
		System.out.println("  " + publicKey.getFormat());
		System.out.println("  Y=" + publicKey.getY());
		System.out.println("  " + publicKey.getParams());
		System.out.println("  " + Hex.encodeHexString(publicKey.getEncoded()));

		// Select a digital signature algorithm
		Signature DSA = Signature.getInstance("SHA256WITHDSA");

		// Initialize it with the private key and the random number generator
		DSA.initSign(privateKey, secRnd);
		
		// This is the message we're going to sign
		String message = new String("Hello Block Week 2019!");
		
		// Set the message bytes into the signature algorithm
		DSA.update(message.getBytes());
		
		// And sign it, getting the signature
		byte[] ECSignature = DSA.sign();

		System.out.println("Message Signature: ");
		System.out.println(Hex.encodeHex(ECSignature));

		// Initialize the ECDSA algorithm for verification
		DSA.initVerify(publicKey);
		
		// Set the message bytes into the signature algorithm
		DSA.update(message.getBytes());
		
		// And verify it
		boolean isSignatureVerified = DSA.verify(ECSignature);
		
		System.out.println("Signature is verified: " + isSignatureVerified);
	}

	
	public static void main(String[] args) throws NoSuchAlgorithmException, InvalidAlgorithmParameterException, InvalidKeyException, SignatureException {

		// Print a list of all security providers and the services they provide
		for (Provider prov : Security.getProviders()) {
			System.out.println("Provider: " + prov.getName());
			for (Service service : prov.getServices()) {
				System.out.println("  " + service);
			}
		}
		
		// Print a list of available random number generators
		System.out.println("Random number generators:");
		for (String algorithm : Security.getAlgorithms("SecureRandom")) {
			System.out.println("  "+algorithm);
		}
		
		// Choose the SHA1 Pseudo-Random-Number-Generator
		secRnd = SecureRandom.getInstance("SHA1PRNG");
		
		// Print a list of available key generator algorithms
		System.out.println("Key pair generators:");
		for (String algorithm : Security.getAlgorithms("KeyPairGenerator")) {
			System.out.println("  "+algorithm);
		}

		ellipticCurvesDemo();

		primeFactorizationDemo();
	}

}
