# Function to generate empirical normally distributed data
#
# IMPORTANT: In contrast to rnorm, this takes the variance, not the standard deviation
#            as input
#
rnorm.emp <- function(nobs, mean=0, var=1) {
  r <- mvrnorm(nobs, matrix(mean, nrow=1, ncol=1), matrix(var, nrow=1, ncol=1), empirical=TRUE)
  r
}

# Function to generate a block of indicators with given indicator
# covariance and the composite with additional error variance
#
# Result is a list of 2 elements, the composite, and the indicator data
#
# Assumption: loadings of 1
#
generate.formative <- function(nobs, nind, indcov, errorvar, name) {
  mu <- rep(0, nind)
  sigma <- matrix(indcov, nrow=nind, ncol=nind)
  diag(sigma) <- rep(1, nind)
  x.dat <- as.data.frame(mvrnorm(nobs, mu, sigma, empirical=TRUE))

  colnames(x.dat) <- paste(name, 1:nind, sep="")
#  message(cov(x.dat))

  comp.z <- rowSums(x.dat)/sqrt(var(rowSums(x.dat)))
#  message(var(comp.z))

  comp.1 <- comp.z * sqrt(1-errorvar)
#  message(var(comp.1))

  if (errorvar > 0) {
    zeta <- rnorm.emp(nobs, 0, errorvar)
  } else {
    zeta <- rep(0, nobs)
  }
#  message(var(zeta))

  comp.2 <- comp.1 + zeta
#  message(var(comp.2))

  comp.df <- as.data.frame(comp.2)
  colnames(comp.df) <- paste(name)

#  message(cov(cbind(x.dat, comp.df)))

  result <- list(length=2)
  result[[1]] <- comp.df
  result[[2]] <- x.dat
  result
}

# Function to make reflective indicators of a single 
# latent variable with additional error variance
# 
# Result is a data frame with the indicator data
#
# Assumption: loadings of 1
#
make.indicators <- function(latent, nind, errorvar) {
  result <- latent
  nobs <- nrow(latent)
  for (i in 1:nind) {
     eps <- rnorm.emp(nobs, 0, errorvar)
# message(var(eps))
     ind <- latent + eps
# message(var(ind))
     ind.z <- ind/sqrt(var(ind))
# message(var(ind.z))
     result <- cbind(result, ind.z)
  }
  result <- result[,-1]
  colnames(result) <- paste(colnames(latent)[1], 1:nind, sep="")
  result
}


# Function to create the data for model 3
# with given number of indicators, indicator covariance for the formative blocks
# and error variance on all endogenous variables
#
# Assumption: Structural path coefficients are 0.8
#
# Returns a list of 2 elements, the exogenous data frame, and the endoenous data frame
#
model3.data <- function(nobs, nind, indcov, errorvar) {
  comp.1 <- generate.formative(nobs, nind, indcov, errorvar, "a")
  comp.2 <- generate.formative(nobs, nind, indcov, errorvar, "b")
  comp.3 <- generate.formative(nobs, nind, indcov, errorvar, "c")

  A <- comp.1[[1]]
  A.ind <- comp.1[[2]]
  B <- comp.2[[1]]
  B.ind <- comp.2[[2]]
  C <- comp.3[[1]]
  C.ind <- comp.3[[2]]

  zeta.1 <- rnorm.emp(nobs, 0, 0.1)
#  message(var(zeta.1))
  K <- (0.8*A + 0.8*B + zeta.1)/sqrt(var(0.8*A + 0.8*B + zeta.1))
  colnames(K) <- "k"
#  message(var(K))

  zeta.2 <- rnorm.emp(nobs, 0, 0.1)
#  message(var(zeta.2))
  L <- (0.8*B + 0.8*C + zeta.2)/sqrt(var(0.8*B + 0.8*C + zeta.2))
  colnames(L) <- "l"
#  message(var(L))

  K.ind <- make.indicators(K, nind, 0.1)
  L.ind <- make.indicators(L, nind, 0.1)

  zeta.3 <- rnorm.emp(nobs, 0, 0.1)
#  message(var(zeta.3))
  X <- (0.8*K + 0.8*A + zeta.3)/sqrt(var(0.8*A + zeta.3))
  colnames(X) <- "x"
#  message(var(X))

  zeta.4 <- rnorm.emp(nobs, 0, 0.1)
#  message(var(zeta.4))
  Y <- (0.8*K + 0.8*L + 0.8*B + zeta.4)/sqrt(var(0.8*K + 0.8*L + zeta.4))
  colnames(Y) <- "y"
#  message(var(Y))

  zeta.5 <- rnorm.emp(nobs, 0, 0.1)
#  message(var(zeta.5))
  Z <- (0.8*L + 0.8*C + zeta.5)/sqrt(var(0.8*L + zeta.5))
  colnames(Z) <- "z"
#  message(var(Z))

  X.ind <- make.indicators(X, nind, 0.1)
  Y.ind <- make.indicators(Y, nind, 0.1)
  Z.ind <- make.indicators(Z, nind, 0.1)

  Y.mat <- as.matrix(cbind(K.ind, L.ind, X.ind, Y.ind, Z.ind))
  X.mat <- as.matrix(cbind(A.ind, B.ind, C.ind))

  r <- list(X.mat, Y.mat)
  r
}

generate <- function(model, nobs, nind, indcov, errorvar) {
   if (model == 0)
      data <- model0.data(nobs, nind, indcov, errorvar)
   if (model == 1)
      data <- model1.data(nobs, nind, indcov, errorvar)
   if (model == 2)
      data <- model2.data(nobs, nind, indcov, errorvar)
   if (model == 3)
      data <- model3.data(nobs, nind, indcov, errorvar)

   data
}


