import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.sql.ResultSet;
import java.io.FileNotFoundException;
import java.io.PrintStream;
import java.io.FileOutputStream;
import simpack.measure.string.MongeElkan;
import simpack.accessor.string.StringAccessor;

public class Match {

	// parameters for the SAIM algorithm
	private final float L = 1f; // lag factor
	private final float eps = 0.001f;
	private final float weightObjAtt = 1f; // weights for influence of activations
	private final float weightObjObj = 1f;
	private final float weightObjRole = 1f;
	private final float weightAttAtt = 1f;
	private final float weightRoleRole = 1f;
	
	// data structures for objects, attributes, roles and their match nodes
	private int nobja = 0;
	private int nobjb = 0;
	private int natta = 0;
	private int nattb = 0;
	private int nrolesa = 0;
	private int nrolesb = 0;
	
	// Names and Other Information
	private String[] ObjectNamesa;
	private String[] ObjectNamesb;
	private String[] AttNamesa;
	private String[] AttTypesa;
	private String[] AttNamesb;
	private String[] AttTypesb;
	private String[] RoleNamesa;
	private String[] RoleRulesDela;
	private String[] RoleRulesUpda;
	private String[] RoleNamesb;
	private String[] RoleRulesDelb;
	private String[] RoleRulesUpdb;
	
	// Indicates which Object owns which Attributes (for schemas a and b)
	private boolean[][] ObjectAttributesa;
	private boolean[][] ObjectAttributesb;
	// Indicates which Object participates in which Roles (for schemas a and b)
	private byte[][] ObjectRolesa;
	private byte[][] ObjectRolesb;
	
	// Initial/Static similarities based on surface similarity metrics
	private float[][] attributeSimilarity;
	// Initial/Static similarities based on surface similarity metrics
	private float[][] roleSimilarity;
	
	private int niter = 1;
	private final int nitermax = 100;
	private float maxp = 0;
	
	// Object-Object Match Nodes for Iteration 1
	private float[][] obj1 = new float[nobja][nobjb];
	// Attribute-Attribute Match Nodes for Iteration 1
	private float[][] att1 = new float[natta][nattb];
	// Role-Role Match Nodes for Iteration 1
	private float[][] roles1 = new float[nrolesa][nrolesb];

	// Object-Object Match Nodes for Iteration 2
	private float[][] obj2 = new float[nobja][nobjb];
	// Attribute-Attribute Match Nodes for Iteration 2
	private float[][] att2 = new float[natta][nattb];
	// Role-Role Match Nodes for Iteration 2
	private float[][] roles2 = new float[nrolesa][nrolesb];
	
	// The final similarity of objects
	private float[][] objectSim;
	
	private PrintStream ps;
	
	public Match() { }
	
	public void main(String conn1, String conn2, String filename) {
		boolean conv = false;
		initialize(conn1, conn2, filename);
		calculateObjectSim();
		while (!conv) {
			print();
			update_obj();
			update_att();
			update_roles();
			calculateObjectSim();
			conv = converged();
			swap();
			niter++;
		}	
		print();
		ps.close();
		mappingProb(new int[Math.max(nobja,  nobjb)], 0, 0);
	}
	
	private void mappingProb(int[] c, int l, float p) {
		if (nobja <= nobjb) {
			if (l == nobja) {
				if (p >= maxp) {
					maxp = p;
					printMappinga(c, p);
				}
			}
			else {
				for (int i=0; i<nobjb; i++) {
					if (!contains(c, i+1)) {
						c[l] = i+1;
						mappingProb(c.clone(), l+1, p + obj1[l][i]);
//						mappingProb(c.clone(), l+1, p + objectSim[l][i]);
					} 
				}
			}
		} else {
			if (l == nobjb) {
				if (p >= maxp) {
					maxp = p;
					printMappingb(c, p);
				}
			}
			else {
				for (int i=0; i<nobja; i++) {
					if (!contains(c, i+1)) {
						c[l] = i+1;
						mappingProb(c.clone(),l+1, p + obj1[i][l]);
//						mappingProb(c.clone(),l+1, p + objectSim[i][l]);
					} 
				}
			}
		}
	}
	
	private void printMappinga(int[] c, float p) {
		System.out.println("Mapping with p = " + String.valueOf(p));
		for (int i=0; i<nobja; i++)
			System.out.println(ObjectNamesa[i] + " --- " + ObjectNamesb[c[i]-1]);
		System.out.println();
	}

	private void printMappingb(int[] c, float p) {
		System.out.println("Mapping with p = " + String.valueOf(p));
		for (int i=0; i<nobjb; i++)
			System.out.println(ObjectNamesa[c[i]-1] + " --- " + ObjectNamesb[i]);
		System.out.println();
	}

	
	private boolean contains(int[] c, int l) {
		for (int i=0; i<c.length; i++) {
			if (c[i] == l) return true;
		}
		return false;
	}
	
	private void calculateObjectSim() {
		objectSim = new float[nobja][nobjb];
		for (int i=0; i<nobja; i++) {
			for (int j=0; j<nobjb; j++) {
				float n=0; float d=0;
				for (int k=0; k<natta; k++) {
					for (int l=0; l<nattb; l++) {
						if (ObjectAttributesa[i][k] && ObjectAttributesb[j][l]) {
							n += att1[k][l]*attributeSimilarity[k][l];
							d += att1[k][l];
						}
					}
				}
				objectSim[i][j] = n/d;
			}
		}
	}
		
	private void print() {
		System.out.println("iter " + String.valueOf(niter));
		System.out.println("Object-Object Node Activation");
		System.out.print("            ");
		for (int j=0; j<nobjb; j++) {
			System.out.format("%1$-11S", ObjectNamesb[j].substring(0, Math.min(ObjectNamesb[j].length(), 9))); 
		}
		System.out.println();
		for (int i=0; i<nobja; i++) {
			System.out.format("%1$-11S ", ObjectNamesa[i].substring(0, Math.min(ObjectNamesa[i].length(), 9))); 
			for (int j=0; j<nobjb; j++)
				System.out.format("%1$10.8f ", obj1[i][j]); 
			System.out.println();
		}
		System.out.println();
		System.out.println("Object-Object Similarity");
		System.out.print("            ");
		for (int j=0; j<nobjb; j++) {
			System.out.format("%1$-11S", ObjectNamesb[j].substring(0, Math.min(ObjectNamesb[j].length(), 9))); 
		}
		System.out.println();
		for (int i=0; i<nobja; i++) {
			System.out.format("%1$-11S ", ObjectNamesa[i].substring(0, Math.min(ObjectNamesa[i].length(), 9))); 
			for (int j=0; j<nobjb; j++)
				System.out.format("%1$10.8f ", objectSim[i][j]); 
			System.out.println();
		}
		System.out.println();
		System.out.println("Feature-Feature Node Activation");
		System.out.print("           ");
		for (int j=0; j<nattb; j++) {
			System.out.format("%1$-11S", AttNamesb[j].substring(0, Math.min(AttNamesb[j].length(), 9))); 
		}
		System.out.println();
		for (int i=0; i<natta; i++) {
			System.out.format("%1$-11S", AttNamesa[i].substring(0, Math.min(AttNamesa[i].length(), 9))); 
			for (int j=0; j<nattb; j++)
				System.out.format("%1$10.8f ", att1[i][j]); 
			System.out.println();
		}
		System.out.println();
		System.out.println("Feature-Feature Similarity");
		System.out.print("           ");
		for (int j=0; j<nattb; j++) {
			System.out.format("%1$-11S", AttNamesb[j].substring(0, Math.min(AttNamesb[j].length(), 9))); 
		}
		System.out.println();
		for (int i=0; i<natta; i++) {
			System.out.format("%1$-11S", AttNamesa[i].substring(0, Math.min(AttNamesa[i].length(), 9))); 
			for (int j=0; j<nattb; j++)
				System.out.format("%1$10.8f ", att1[i][j]*attributeSimilarity[i][j]); 
			System.out.println();
		}
		System.out.println();
		System.out.println("Role-Role Node Activation");
		System.out.print("           ");
		for (int j=0; j<nrolesb; j++) {
			System.out.format("%1$-11S", RoleNamesb[j].substring(0, Math.min(RoleNamesb[j].length(), 9))); 
		}
		System.out.println();
		for (int i=0; i<nrolesa; i++) {
			System.out.format("%1$-11S", RoleNamesa[i].substring(0, Math.min(RoleNamesa[i].length(), 9))); 
			for (int j=0; j<nrolesb; j++)
				System.out.format("%1$10.8f ", roles1[i][j]); 
			System.out.println();
		}
		System.out.println();
		System.out.println("Role-Role Similarity");
		System.out.print("           ");
		for (int j=0; j<nrolesb; j++) {
			System.out.format("%1$-11S", RoleNamesb[j].substring(0, Math.min(RoleNamesb[j].length(), 9))); 
		}
		System.out.println();
		for (int i=0; i<nrolesa; i++) {
			System.out.format("%1$-11S", RoleNamesa[i].substring(0, Math.min(RoleNamesa[i].length(), 9))); 
			for (int j=0; j<nrolesb; j++)
				System.out.format("%1$10.8f ", roles1[i][j]*roleSimilarity[i][j]); 
			System.out.println();
		}
		System.out.println("===================");

		ps.print(String.valueOf(niter) + ",");
		for (int i=0; i<nobja; i++) {
			for (int j=0; j<nobjb; j++)
				ps.format("%1$10.8f, ", obj1[i][j]);
		}
		for (int i=0; i<natta; i++) {
			for (int j=0; j<nattb; j++)
				ps.format("%1$10.8f,", att1[i][j]); 
		}
		for (int i=0; i<natta; i++) {
			for (int j=0; j<nattb; j++)
				ps.format("%1$10.8f,", att1[i][j]*attributeSimilarity[i][j]); 
		}
		for (int i=0; i<nrolesa; i++) {
			for (int j=0; j<nrolesb; j++)
				ps.format("%1$10.8f,", roles1[i][j]); 
		}
		ps.println();
}
	
	private void initialize (String conn1, String conn2, String filename) {
		Connection conn = null;
		ResultSet rs1, rs2, rs3;

		try {
			Class.forName("org.postgresql.Driver"); 
		} catch (ClassNotFoundException e) {
			return;
		}

		try {
		    conn = DriverManager.getConnection(conn1);
		    rs1 = conn.createStatement().executeQuery("select count(*) as cnt from information_schema.tables where table_schema = 'public'");
		    if (rs1.next()) {
		    	nobja = rs1.getInt("cnt");
		    }
		    rs1.close();
		    rs1 = conn.createStatement().executeQuery("select count(*) as cnt from information_schema.columns where table_schema = 'public'");
		    if (rs1.next()) {
		    	natta = rs1.getInt("cnt");
		    }
		    rs1.close();
		    rs1 = conn.createStatement().executeQuery("select count(*) as cnt from information_schema.referential_constraints where constraint_schema = 'public'");
		    if (rs1.next()) {
		    	nrolesa = rs1.getInt("cnt");
		    }
		    rs1.close();
		    
		    // Set up data structures to hold info
		    ObjectNamesa = new String[nobja];
		    AttNamesa = new String[natta];
		    AttTypesa = new String[natta];
		    RoleNamesa = new String[nrolesa];
		    RoleRulesUpda = new String[nrolesa];
		    RoleRulesDela = new String[nrolesa];
		    ObjectAttributesa = new boolean[nobja][natta];
		    ObjectRolesa = new byte[nobja][nrolesa];
		    
		    int t = 0, a = 0, r = 0;
		    rs1 = conn.createStatement().executeQuery("select * from information_schema.tables where table_schema = 'public'");
		    while (rs1.next()) {
		    	ObjectNamesa[t] = rs1.getString("TABLE_NAME");
		    	rs2 = conn.createStatement().executeQuery("Select * from information_schema.columns where table_schema = 'public' and table_name = '"+ObjectNamesa[t]+"'");
		    	while (rs2.next()) {
		    		AttNamesa[a] = rs2.getString("COLUMN_NAME");
		    		AttTypesa[a] = rs2.getString("DATA_TYPE");
		    		ObjectAttributesa[t][a] = true;
		    		a++;
		    	}
		    	r=0;
		    	rs3 = conn.createStatement().executeQuery(
		    			"select distinct tc1.table_name as table1, rc.constraint_name as role, rc.update_rule as update, rc.delete_rule as delete, tc2.table_name as table2 " + 
		    			"from information_schema.table_constraints as tc1, " +
		    			"information_schema.referential_constraints as rc, " +
		    			"information_schema.table_constraints as tc2 where " +
		    			"tc1.constraint_name = rc.constraint_name and " +
		    			"tc2.constraint_name = rc.unique_constraint_name and " +
		    			"rc.constraint_catalog = rc.unique_constraint_catalog and " +
		    			"rc.constraint_catalog = tc1.constraint_catalog and " +
		    			"rc.constraint_catalog = tc2.constraint_catalog and " +
		    			"tc1.constraint_catalog = tc1.table_catalog and " +
		    			"tc2.constraint_catalog = tc2.table_catalog and " +
		    			"tc1.constraint_schema = 'public' and " +
		    			"tc2.constraint_schema = 'public' and " +
		    			"tc1.table_schema = 'public' and " +
		    			"tc2.table_schema = 'public' and " +
		    			"rc.constraint_schema = 'public' and " +
		    			"rc.unique_constraint_schema = 'public' " +
		    			"order by rc.constraint_name asc");
		    	while (rs3.next()) {
		    		RoleNamesa[r] = rs3.getString("role");
		    		RoleRulesUpda[r] = rs3.getString("update");
		    		RoleRulesDela[r] = rs3.getString("delete");
		    		if (rs3.getString("table1").equals(ObjectNamesa[t]))
		    			ObjectRolesa[t][r] = 1;
		    		else if (rs3.getString("table2").equals(ObjectNamesa[t])) 
		    			ObjectRolesa[t][r] = 2;
		    		else
		    			ObjectRolesa[t][r] = 0;
		    		r++;
		    	}	
		    	t++;
		    }
		    conn.close();
		    conn = DriverManager.getConnection(conn2);
		    rs1 = conn.createStatement().executeQuery("select count(*) as cnt from information_schema.tables where table_schema = 'public'");
		    if (rs1.next()) {
		    	nobjb = rs1.getInt("cnt");
		    }
		    rs1.close();
		    rs1 = conn.createStatement().executeQuery("select count(*) as cnt from information_schema.columns where table_schema = 'public'");
		    if (rs1.next()) {
		    	nattb = rs1.getInt("cnt");
		    }
		    rs1.close();
		    rs1 = conn.createStatement().executeQuery("select count(*) as cnt from information_schema.referential_constraints where constraint_schema = 'public'");
		    if (rs1.next()) {
		    	nrolesb = rs1.getInt("cnt");
		    }
		    rs1.close();
		    
		    // Set up data structures to hold info
		    ObjectNamesb = new String[nobjb];
		    AttNamesb = new String[nattb];
		    AttTypesb = new String[nattb];
		    RoleNamesb = new String[nrolesb];
		    RoleRulesUpdb = new String[nrolesb];
		    RoleRulesDelb = new String[nrolesb];
		    ObjectAttributesb = new boolean[nobjb][nattb];
		    ObjectRolesb = new byte[nobjb][nrolesb];
		    
		    t=0; a=0; r=0;
		    rs1 = conn.createStatement().executeQuery("select * from information_schema.tables where table_schema = 'public'");
		    while (rs1.next()) {
		    	ObjectNamesb[t] = rs1.getString("TABLE_NAME");
		    	rs2 = conn.createStatement().executeQuery("Select * from information_schema.columns where table_schema = 'public' and table_name = '"+ObjectNamesb[t]+"'");
		    	while (rs2.next()) {
		    		AttNamesb[a] = rs2.getString("COLUMN_NAME");
		    		AttTypesb[a] = rs2.getString("DATA_TYPE");
		    		ObjectAttributesb[t][a] = true;
		    		a++;
		    	}
		    	r = 0;
		    	rs3 = conn.createStatement().executeQuery(
		    			"select distinct tc1.table_name as table1, rc.constraint_name as role, rc.update_rule as update, rc.delete_rule as delete, tc2.table_name as table2 " + 
		    			"from information_schema.table_constraints as tc1, " +
		    			"information_schema.referential_constraints as rc, " +
		    			"information_schema.table_constraints as tc2 where " +
		    			"tc1.constraint_name = rc.constraint_name and " +
		    			"tc2.constraint_name = rc.unique_constraint_name and " +
		    			"rc.constraint_catalog = rc.unique_constraint_catalog and " +
		    			"rc.constraint_catalog = tc1.constraint_catalog and " +
		    			"rc.constraint_catalog = tc2.constraint_catalog and " +
		    			"tc1.constraint_catalog = tc1.table_catalog and " +
		    			"tc2.constraint_catalog = tc2.table_catalog and " +
		    			"tc1.constraint_schema = 'public' and " +
		    			"tc2.constraint_schema = 'public' and " +
		    			"tc1.table_schema = 'public' and " +
		    			"tc2.table_schema = 'public' and " +
		    			"rc.constraint_schema = 'public' and " +
		    			"rc.unique_constraint_schema = 'public' " +
		    			"order by rc.constraint_name asc");
		    	while (rs3.next()) {
		    		RoleNamesb[r] = rs3.getString("role");
		    		RoleRulesUpdb[r] = rs3.getString("update");
		    		RoleRulesDelb[r] = rs3.getString("delete");
		    		if (rs3.getString("table1").equals(ObjectNamesb[t]))
		    			ObjectRolesb[t][r] = 1; // source of FK
		    		else if (rs3.getString("table2").equals(ObjectNamesb[t])) 
		    			ObjectRolesb[t][r] = 2; // target of FK
		    		else
		    			ObjectRolesb[t][r] = 0;
		    		r++;
		    	}	
		    	t++;
		    }
		    conn.close();
		    
		} catch (SQLException ex) {
		    System.out.println("SQLException: " + ex.getMessage());
		    System.out.println("SQLState: " + ex.getSQLState());
		    System.out.println("VendorError: " + ex.getErrorCode());
		}
		initialize_role_similarity();
		initialize_attr_similarity();
		initialize_activations();

		try {
			ps = new PrintStream(new FileOutputStream(filename));
			ps.print("iter,");
			for (int i=0; i<nobja; i++)
				for (int j=0; j<nobjb; j++) {
					ps.print(ObjectNamesa[i].substring(0, Math.min(ObjectNamesa[i].length(), 9))+ObjectNamesb[j].substring(0, Math.min(ObjectNamesb[j].length(), 9))+",");
				}
			for (int i=0; i<natta; i++)
				for (int j=0; j<nattb; j++) {
					ps.print(AttNamesa[i].substring(0, Math.min(AttNamesa[i].length(), 9))+AttNamesb[j].substring(0, Math.min(AttNamesb[j].length(), 9))+",");
				}
			for (int i=0; i<natta; i++)
				for (int j=0; j<nattb; j++) {
					ps.print("Sim"+ AttNamesa[i].substring(0, Math.min(AttNamesa[i].length(), 9))+AttNamesb[j].substring(0, Math.min(AttNamesb[j].length(), 9))+",");
				}
			for (int i=0; i<nrolesa; i++)
				for (int j=0; j<nrolesb; j++) {
					ps.print(RoleNamesa[i].substring(0, Math.min(RoleNamesa[i].length(), 9))+RoleNamesb[j].substring(0, Math.min(RoleNamesb[j].length(), 9))+",");
				}
			ps.println();
		} catch (FileNotFoundException e1) {
			e1.printStackTrace();
		}
		
}
	
	private void initialize_role_similarity() {
		roleSimilarity = new float[nrolesa][nrolesb];
		for (int i=0; i < nrolesa; i++) 
			for (int j=0; j < nrolesb; j++)
				roleSimilarity[i][j] = calcRoleSimilarity(RoleNamesa[i], RoleNamesb[j], RoleRulesDela[i], RoleRulesDelb[j], RoleRulesUpda[i], RoleRulesUpdb[j]);
	}
	
	private float StringSimilarity(String name1, String name2) {
		return new MongeElkan(new StringAccessor(name1), new StringAccessor(name2)).getSimilarity().floatValue();
	}
	
	private float calcRoleSimilarity(String name1, String name2, String del1, String del2, String upd1, String upd2) {
		float mes = StringSimilarity(name1, name2);
		return .2f*(del1.equals(del2)?1f:0f) + .2f*(upd1.equals(upd2)?1f:0f) + .6f*mes;
	}

	private void initialize_attr_similarity() {
		attributeSimilarity = new float[natta][nattb];
		for (int i=0; i < natta; i++) 
			for (int j=0; j < nattb; j++)
				attributeSimilarity[i][j] = calcAttSimilarity(AttNamesa[i], AttNamesb[j], AttTypesa[i], AttTypesb[j]);
	}
	
	private float calcAttSimilarity(String name1, String name2, String type1, String type2) {
		if (kindOfType(type1).equals(kindOfType(type2)))
			return (StringSimilarity(name1, name2));
		else
			return 0.8f*(StringSimilarity(name1, name2));
	}

	private String kindOfType(String type1) {
		if (type1.equals("integer") || type1.equals("smallint") || type1.equals("bigint") || type1.equals("serial") || type1.equals("bigserial")) return "integer";
		if (type1.equals("decimal") || type1.equals("numeric") || type1.equals("real") || type1.equals("double precision")) return "decimal";
		if (type1.equals("char") || type1.equals("varchar") || type1.equals("text") || type1.equals("character varying") || type1.equals("character")) return "character";
		if (type1.equals("timestamp") || type1.equals("date") || type1.equals("time") || type1.equals("interval")) return "temporal";
		return type1;
	}

	private void initialize_activations() {
		obj1 = new float[nobja][nobjb];
		att1 = new float[natta][nattb];
		roles1 = new float[nrolesa][nrolesb];

		obj2 = new float[nobja][nobjb];
		att2 = new float[natta][nattb];
		roles2 = new float[nrolesa][nrolesb];
		
		for (int i=0; i < nobja; i++)
			for (int j=0; j < nobjb; j++) {
				obj1[i][j] = 0.5f;
				obj2[i][j] = 0.5f;
			}		
		for (int i=0; i < natta; i++)
			for (int j=0; j < nattb; j++) {
				att1[i][j] = attributeSimilarity[i][j];
				att2[i][j] = attributeSimilarity[i][j];
			}
		for (int i=0; i < nrolesa; i++)
			for (int j=0; j < nrolesb; j++) {
				roles1[i][j] = roleSimilarity[i][j];
				roles2[i][j] = roleSimilarity[i][j];
			}
	}
	
	// Check for convergence
	private boolean converged() {
		for (int i=0; i<nobja; i++)
			for (int j=0; j<nobjb; j++)
				if (Math.abs(obj1[i][j]-obj2[i][j]) > eps) return false;
		for (int i=0; i<natta; i++)
			for (int j=0; j<nattb; j++)
				if (Math.abs(att1[i][j]-att2[i][j]) > eps) return false;
		for (int i=0; i<nrolesa; i++)
			for (int j=0; j<nrolesb; j++)
				if (Math.abs(roles1[i][j]-roles2[i][j]) > eps) return false;
		if (niter >= nitermax) return true;
		return true;
	}
	
	// Swap the two iterations
	private void swap() {
		Object tmp;
		tmp = obj1; obj1 = obj2; obj2 = (float[][]) tmp;
		tmp = att1; att1 = att2; att2 = (float[][]) tmp;
		tmp = roles1; roles1 = roles2; roles2 = (float[][]) tmp;
	}
	
	// Check whether any Node-Node connection is consistent or inconsistent
	private boolean isConsistent(char a, int i, int j, char b, int k, int l) {
		boolean c = false;
		// Mapping of objects i to j and k to l
		if (a == 'o' && b == 'o') c = (i != k && j != l);
		// Mapping of objects i to j and attributes k and l
		if (a == 'o' && b == 'a') c = (ObjectAttributesa[i][k] == ObjectAttributesb[j][l]); // or "&&" ??
		// Mapping of objects i to j and roles k and l
		if (a == 'o' && b == 'r') c = (ObjectRolesa[i][k] == ObjectRolesb[j][l]);

		// Mapping of attributes i to j and k to l
		if (a == 'a' && b == 'a') c = (i != k && j != l);
		// Mapping of attributes i to j and objects k to l
		if (a == 'a' && b == 'o') c = (ObjectAttributesa[k][i] == ObjectAttributesb[l][j]); // or "&&" ??
		
		// Mapping of roles i to j and k to l
		if (a == 'r' && b == 'r') {
			if (i == k || j == l) c = false;
			else {
				int obji1 = getObjRole(i, 1, 'a');
				int obji2 = getObjRole(i, 2, 'a');
				int objj1 = getObjRole(j, 1, 'b');
				int objj2 = getObjRole(j, 2, 'b');
				int objk1 = getObjRole(i, 1, 'a');
				int objk2 = getObjRole(i, 2, 'a');
				int objl1 = getObjRole(j, 1, 'b');
				int objl2 = getObjRole(j, 2, 'b');
				c = isConsistent('o', obji1, objj1, 'o', obji2, objj2) &&
					isConsistent('o', obji1, objj1, 'o', objk1, objl1) &&
					isConsistent('o', obji1, objj1, 'o', objk2, objl2) &&
					isConsistent('o', obji2, objj2, 'o', objk1, objl1) &&
					isConsistent('o', obji2, objj2, 'o', objk2, objl2) &&
					isConsistent('o', objk1, objl1, 'o', objk2, objl2);
			}
		}
		// Mapping of roles i to j and objects k to l
		if (a == 'r' && b == 'o') c = (ObjectRolesa[k][i] == ObjectRolesb[l][j]);
		
		return c;
	}
	
	private int getObjRole(int role, int pos, char side) {
		if (side == 'a')
			for (int i=0; i<nobja; i++)
				if (ObjectRolesa[i][role] == pos) return i;
		if (side == 'b')
			for (int i=0; i<nobjb; i++)
				if (ObjectRolesb[i][role] == pos) return i;
		return 0;
	}
	
	// Calculate the contribution for excitatory (consistent) node-node connections
	private float calcExcitContrib(float Ai, float Aj) {
		if (Aj > .5)
			return (Ai + (1f - Ai)*(Aj - 0.5f));
		else
			return (Ai - Ai * (0.5f - Aj));
	}

	// Calculate the contribution for inhibitory (inconsistent) node-node connections
	private float calcInhibContrib(float Ai, float Aj) {
		if ( (1 - Aj) > .5)
			return (Ai + (1f - Ai)*((1 - Aj) - 0.5f));
		else
			return (Ai - Ai * (0.5f - (1 - Aj)));
	}

	// Update Object-Object Match Node similarity
	private float calculateObjUpdate(int i, int j) {
		float M = 0f;
		
		// Other object nodes
		for (int k=0; k < nobja; k++)
			for (int l=0; l < nobjb; l++)
				if (k != i || l != j)
					if (isConsistent('o', i, j, 'o', k, l)) {
						M += weightObjObj * calcExcitContrib(obj1[i][j], obj1[k][l]);
					} else {
						M += weightObjObj * calcInhibContrib(obj1[i][j], obj1[k][l]);
					}

		// Attribute nodes
		for (int k=0; k < natta; k++)
			for (int l=0; l < nattb; l++)
				if (isConsistent('o', i, j, 'a', k, l)) {
					M += weightObjAtt * calcExcitContrib(obj1[i][j], att1[k][l]);
				} else {
					M += weightObjAtt * calcInhibContrib(obj1[i][j], att1[k][l]);
				}

		// Role nodes
		for (int k=0; k < nrolesa; k++)
			for (int l=0; l < nrolesb; l++)
				if (isConsistent('o', i, j, 'r', k, l)) {
					M += weightObjRole * calcExcitContrib(obj1[i][j], roles1[k][l]);
				} else {
					M += weightObjRole * calcInhibContrib(obj1[i][j], roles1[k][l]);					
				}
		return M;
	}

	// Update Attribute-Attribute Match Node similarity
	private float calculateAttUpdate(int i, int j) {
		float M = 0f;
		float contrib = 0f;
		
		// Other attribute nodes
		for (int k=0; k < natta; k++)
			for (int l=0; l < nattb; l++)
				if (k != i || l != j)
					if (isConsistent('a', i, j, 'a', k, l)) {
						contrib = calcExcitContrib(att1[i][j], att1[k][l]);
						M += weightAttAtt * contrib;
					} else {
						contrib = calcInhibContrib(att1[i][j], att1[k][l]);
						M += weightAttAtt * contrib;
					}

		// Object nodes
		for (int k=0; k < nobja; k++)
			for (int l=0; l < nobjb; l++)
				if (isConsistent('a', i, j, 'o', k, l)) {
					contrib = calcExcitContrib(att1[i][j], obj1[k][l]);
					M += weightObjAtt * contrib;
				} else {
					contrib = calcInhibContrib(att1[i][j], obj1[k][l]);
					M += weightObjAtt * contrib;					
				}

		// Role nodes are NOT CONNECTED TO attribute nodes!!
		
		return M;
	}
	
	// Update Role-Role Match Node similarity
	private float calculateRoleUpdate(int i, int j) {
		float M = 0f;
		
		// Other role nodes
		for (int k=0; k < nrolesa; k++)
			for (int l=0; l < nrolesb; l++)
				if (k != i || l != j)
					if (isConsistent('r', i, j, 'r', k, l)) {
						M += weightRoleRole * calcExcitContrib(roles1[i][j], roles1[k][l]);
					} else {
						M += weightRoleRole * calcInhibContrib(roles1[i][j], roles1[k][l]);					
					}

		// Object nodes
		for (int k=0; k < nobja; k++)
			for (int l=0; l < nobjb; l++)
				if (isConsistent('r', i, j, 'o', k, l)) {
					M += weightObjRole * calcExcitContrib(roles1[i][j], obj1[k][l]);
				} else {
					M += weightObjRole * calcInhibContrib(roles1[i][j], obj1[k][l]);										
				}

		// Role nodes are NOT CONNECTED TO attribute nodes!!

		return M;
	}
	
	// Update Object Mappings
	private void update_obj() {
		float denom = (nobja * nobjb - 1) * weightObjObj + (natta * nattb) * weightObjAtt + (nrolesa * nrolesb) * weightObjRole;
		for(int i=0; i < nobja; i++)
			for (int j=0; j < nobjb; j++) {
				obj2[i][j] = obj1[i][j] * (1 - L) + L * calculateObjUpdate(i, j) / denom;
		}
	}
	
	// Update Attribute Mappings
	private void update_att() {
		float denom = (nobja * nobjb) * weightObjObj + (natta * nattb - 1) * weightObjAtt;
		for(int i=0; i < natta; i++)
			for (int j=0; j<nattb; j++) {
				att2[i][j] = att1[i][j] * (1f - L) + L * calculateAttUpdate(i, j) / denom;
		}
	}

	// UPdate Role Mappings
	private void update_roles() {
		float denom = (nobja * nobjb) * weightObjObj + (nrolesa * nrolesb - 1) * weightObjRole;
		for(int i=0; i < nrolesa; i++)
			for (int j=0; j<nrolesb; j++) {
				roles2[i][j] = roles1[i][j] * (1f - L) + L * calculateRoleUpdate(i, j) / denom;
		}
	}
	

}
