<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform" xmlns:aspectJ='http://www.magicdraw.com/schemas/aspectJ.xmi' xmlns:uml="http://schema.omg.org/spec/UML/2.0" xmlns:xmi="http://schema.omg.org/spec/XMI/2.1" version="1.0">

<!-- This works with MagicDraw 15 -->
<!-- MagicDraw 15 uses UML 2.1.2 and XMI 2.1 -->

	<xsl:output method="text"/>

<!-- Override the built-in template that outputs all text -->
	<xsl:template match="text()|@*">
		<xsl:apply-templates/>
	</xsl:template>

	<xsl:template match="*" priority="-1">
		<xsl:apply-templates/>
	</xsl:template>

	<xsl:template match="profileApplication">
		<xsl:message terminate="no">Model element <xsl:value-of select="../@name"/> applies a profile</xsl:message>
		
		<xsl:if test="@appliedProfile">
			<xsl:variable name="profileID" select="@appliedProfile"/>
			<xsl:if test="//packagedElement[@xmi:type='uml:Profile' and @xmi:id=$profileID and @name='aspectJ']">
				<xsl:message terminate="no">The applied profile is AspectJ. Profile application and profile found in file.</xsl:message>
				<xsl:call-template name="processAspectJ">
					<xsl:with-param name="modelElement" select=".."/>
				</xsl:call-template>
			</xsl:if>
		</xsl:if>
		
		<xsl:if test="appliedProfile[@href]">
			<xsl:variable name="url" select="appliedProfile/@href"/>
			<xsl:variable name="profileID" select="substring-after($url, '#')"/>
			<xsl:if test="document(substring-before($url, '#'))//packagedElement[@xmi:id=$profileID and @name='aspectJ']">
				<xsl:message terminate="no">The applied profile is AspectJ. Profile found in external file: <xsl:value-of select="$url"/></xsl:message>
				<xsl:call-template name="processAspectJ">
					<xsl:with-param name="modelElement" select=".."/>
				</xsl:call-template>
			</xsl:if>
		</xsl:if>
	</xsl:template>
	
	<xsl:template name="processAspectJ">
		<xsl:param name="modelElement"/>
		<xsl:message terminate="no">Processing model element <xsl:value-of select="$modelElement/@name"/> for AspectJ extensions</xsl:message>
		<xsl:variable name="currentID" select="$modelElement/@xmi:id"/>
		<xsl:if test="//aspectJ:CrossCuttingConcern[@base_Package=$currentID]">
			<xsl:text>package </xsl:text><xsl:value-of select="$modelElement/@name"/>;

<xsl:for-each select="$modelElement/packagedElement[@xmi:type='uml:Class' and @xmi:id]">
				<xsl:variable name="currentID" select="@xmi:id"/>
				<xsl:if test="//aspectJ:Aspect[@base_Class=$currentID]">
					<xsl:variable name="aspect" select="//aspectJ:Aspect[@base_Class=$currentID]"/>
					<xsl:if test="$aspect/@isPrivileged='true'">
						<xsl:text>privileged </xsl:text>
					</xsl:if>
					<xsl:text>aspect </xsl:text><xsl:value-of select="@name"/> {
<xsl:call-template name="processDeclareParents">
						<xsl:with-param name="parentlist" select="$aspect/@declaredParents"/>
					</xsl:call-template>
					<xsl:call-template name="processDeclareImplements">
						<xsl:with-param name="implementslist" select="$aspect/@declaredImplements"/>
					</xsl:call-template><xsl:text>
</xsl:text>
					<xsl:for-each select="ownedAttribute[@xmi:type='uml:Property' and @xmi:id]">
						<xsl:message terminate="no">Processing attribute...</xsl:message>
						<xsl:variable name="attributeID" select="@xmi:id"/>
						<xsl:if test="//aspectJ:StaticCrossCuttingFeature[@base_Feature=$attributeID]">
							<xsl:call-template name="CrossCuttingAttribute">
								<xsl:with-param name="attribute" select="."/>
								<xsl:with-param name="ontypes" select="//aspectJ:StaticCrossCuttingFeature[@base_Feature=$attributeID]/@onType"/>
							</xsl:call-template>
						</xsl:if>
						<xsl:if test="//aspectJ:*[@base_StructuralFeature=$attributeID]">
							<xsl:call-template name="ProcessPointCut">
								<xsl:with-param name="attribute" select="."/>
								<xsl:with-param name="pointcut" select="//aspectJ:*[@base_StructuralFeature=$attributeID]"/>
							</xsl:call-template>
						</xsl:if>
					</xsl:for-each>
					<xsl:text>
</xsl:text>
					<xsl:for-each select="ownedOperation[@xmi:type='uml:Operation' and @xmi:id]">
						<xsl:message terminate="no">Processing operation...</xsl:message>
						<xsl:variable name="operationID" select="@xmi:id"/>
						<xsl:if test="//aspectJ:StaticCrossCuttingFeature[@base_Feature=$operationID]">
							<xsl:call-template name="CrossCuttingOperation">
								<xsl:with-param name="operation" select="."/>
								<xsl:with-param name="ontypes" select="//aspectJ:StaticCrossCuttingFeature[@base_Feature=$operationID]/@onType"/>
							</xsl:call-template>
						</xsl:if>
						<xsl:if test="//aspectJ:Advice[@base_BehavioralFeature=$operationID]">
							<xsl:variable name="advice" select="//aspectJ:Advice[@base_BehavioralFeature=$operationID]"/>
							<xsl:call-template name="ProcessAdvice">
								<xsl:with-param name="operation" select="."/>
								<xsl:with-param name="adviceexecution" select="$advice/@adviceExecution"/>
								<xsl:with-param name="pointcutID" select="$advice/@pointCut"/>
							</xsl:call-template>
						</xsl:if>
					</xsl:for-each>
					<xsl:text>}</xsl:text>
				</xsl:if>
			</xsl:for-each>
		</xsl:if>
		<xsl:for-each select="$modelElement/packagedElement[@xmi:type='uml:Package' or @xmi:type='uml:Model']">
			<xsl:call-template name="processAspectJ">
				<xsl:with-param name="modelElement" select="."/>
			</xsl:call-template>
		</xsl:for-each>
	</xsl:template>
		
	<xsl:template name="processDeclareParents">
		<xsl:param name="parentlist"/>
<!-- Do not call with an empty list -->
		<xsl:choose>
			<xsl:when test="contains($parentlist, ' ')">
				<xsl:variable name="head" select="substring-before($parentlist, ' ')"/>
				<xsl:variable name="body" select="substring-after($parentlist, ' ')"/>
				<xsl:variable name="generalization" select="//generalization[@xmi:type='uml:Generalization' and @xmi:id=$head]"/>
				<xsl:variable name="specific" select="$generalization/../@name"/>
				<xsl:variable name="genID" select="$generalization/@general"/>
				<xsl:variable name="general" select="//*[@xmi:id=$genID]/@name"/>
				<xsl:text>  declare parents: </xsl:text><xsl:value-of select="$specific"/><xsl:text> extends </xsl:text><xsl:value-of select="$general"/>;
<xsl:call-template name="processDeclareParents">
					<xsl:with-param name="parentlist" select="$body"/>
				</xsl:call-template>
			</xsl:when>
			<xsl:otherwise>
				<xsl:variable name="generalization" select="//generalization[@xmi:type='uml:Generalization' and @xmi:id=$parentlist]"/>
				<xsl:variable name="specific" select="$generalization/../@name"/>
				<xsl:variable name="genID" select="$generalization/@general"/>
				<xsl:variable name="general" select="//*[@xmi:id=$genID]/@name"/>
				<xsl:text>  declare parents: </xsl:text><xsl:value-of select="$specific"/><xsl:text> extends </xsl:text><xsl:value-of select="$general"/>;
</xsl:otherwise>				
		</xsl:choose>
	</xsl:template>
	
	<xsl:template name="processDeclareImplements">
		<xsl:param name="implementslist"/>
<!-- Do not call with an empty list -->
		<xsl:choose>
			<xsl:when test="contains($implementslist, ' ')">
				<xsl:variable name="head" select="substring-before($implementslist, ' ')"/>
				<xsl:variable name="body" select="substring-after($implementslist, ' ')"/>
	
				<xsl:variable name="realization" select="//interfaceRealization[@xmi:type='uml:InterfaceRealization' and @xmi:id=$head]"/>
				<xsl:variable name="supplierID" select="$realization/supplier/@xmi:idref"/>
				<xsl:variable name="clientID" select="$realization/client/@xmi:idref"/>
				<xsl:variable name="supplier" select="//*[@xmi:id=$supplierID]/@name"/>
				<xsl:variable name="client" select="//*[@xmi:id=$clientID]/@name"/>
				<xsl:text>  declare parents: </xsl:text><xsl:value-of select="$client"/><xsl:text> implements </xsl:text><xsl:value-of select="$supplier"/>;
<xsl:call-template name="processDeclareImplements">
					<xsl:with-param name="implementslist" select="$body"/>
				</xsl:call-template>
			</xsl:when>
			<xsl:otherwise>
				<xsl:variable name="realization" select="//interfaceRealization[@xmi:type='uml:InterfaceRealization' and @xmi:id=$implementslist]"/>
				<xsl:variable name="supplierID" select="$realization/supplier/@xmi:idref"/>
				<xsl:variable name="clientID" select="$realization/client/@xmi:idref"/>
				<xsl:variable name="supplier" select="//*[@xmi:id=$supplierID]/@name"/>
				<xsl:variable name="client" select="//*[@xmi:id=$clientID]/@name"/>
				<xsl:text>  declare parents: </xsl:text><xsl:value-of select="$client"/><xsl:text> implements </xsl:text><xsl:value-of select="$supplier"/>;
</xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	

	<xsl:template name="buildName">
		<xsl:param name="element"/>
		<xsl:param name="name"/>
		<xsl:message terminate="no">Building name for <xsl:value-of select="$element/@name"/> and <xsl:value-of select="$name"/></xsl:message>
		<xsl:choose>
			<xsl:when test="$element/../@xmi:type='uml:Package' or $element/../@xmi:type='uml:Class'">
				<xsl:call-template name="buildName">
					<xsl:with-param name="element" select="$element/.."/>
					<xsl:with-param name="name" select="concat($element/../@name,'.',$name)"/>
				</xsl:call-template>
			</xsl:when>
			<xsl:otherwise>
				<xsl:value-of select="concat($element/@name,'.',$name)"/>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>

	<xsl:template name="writeType">
		<xsl:param name="type"/>
		<xsl:variable name="thetype" select="//*[@xmi:id=$type]"/>
		<xsl:call-template name="buildName">
			<xsl:with-param name="element" select="$thetype/.."/>
			<xsl:with-param name="name" select="$thetype/@name"/>
		</xsl:call-template>
	</xsl:template>

	<xsl:template name="CrossCuttingOperation">
		<xsl:param name="operation"/>
		<xsl:param name="ontypes"/>
		
		<xsl:choose>
			<xsl:when test="contains($ontypes, ' ')">
				<xsl:variable name="head" select="substring-before($ontypes, ' ')"/>
				<xsl:variable name="body" select="substring-after($ontypes, ' ')"/>
				<xsl:call-template name="writeOperation">
					<xsl:with-param name="operation" select="$operation"/>
					<xsl:with-param name="ontype" select="$head"/>
				</xsl:call-template>
				<xsl:call-template name="CrossCuttingOperation">
					<xsl:with-param name="operation" select="$operation"/>
					<xsl:with-param name="ontypes" select="$body"/>
				</xsl:call-template>
			</xsl:when>
			<xsl:otherwise>
				<xsl:call-template name="writeOperation">
					<xsl:with-param name="operation" select="$operation"/>
					<xsl:with-param name="ontype" select="$ontypes"/>
				</xsl:call-template>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>

	<xsl:template name="writeOperation">
		<xsl:param name="operation"/>
		<xsl:param name="ontype"/>

		<xsl:text>  </xsl:text><xsl:value-of select="$operation/@visibility"/><xsl:text> </xsl:text>
		<xsl:choose>
			<xsl:when test="count($operation/ownedParameter[@direction='return'])=1">
				<xsl:choose>
					<xsl:when test="count($operation/ownedParameter[@direction='return']/type)=1">
						<xsl:value-of select="translate(normalize-space(translate(translate($operation/ownedParameter[@direction='return']/type/xmi:Extension/referenceExtension/@referentPath,' ','_'),':',' ')),' ','.')"/>
					</xsl:when>
					<xsl:when test="count($operation/ownedParameter[@direction='return']/@type)=1">
						<xsl:call-template name="writeType"><xsl:with-param name="type" select="$operation/ownedParameter[@direction='return']/@type"/></xsl:call-template>
					</xsl:when>
				</xsl:choose>
				<xsl:text> </xsl:text>
			</xsl:when>
			<xsl:otherwise>
				<xsl:text>void </xsl:text>
			</xsl:otherwise>
		</xsl:choose>
		<xsl:call-template name="buildName">
			<xsl:with-param name="element" select="//*[@xmi:id=$ontype]"/>
			<xsl:with-param name="name" select="$operation/@name"/>			
		</xsl:call-template>
		<xsl:text> (</xsl:text>
		<xsl:for-each select="$operation/ownedParameter[@direction='in' or @direction='inout']">
			<xsl:choose>
				<xsl:when test="count(@type)=1">
					<xsl:call-template name="writeType"><xsl:with-param name="type" select="@type"/></xsl:call-template>
				</xsl:when>
				<xsl:when test="count(type)=1">
					<xsl:value-of select="translate(normalize-space(translate(translate(type/xmi:Extension/referenceExtension/@referentPath,' ','_'),':',' ')),' ','.')"/>
				</xsl:when>
			</xsl:choose>
			<xsl:text> </xsl:text><xsl:value-of select="@name"/>
			<xsl:if test="position() &lt; count(../ownedParameter[@direction='in' or @direction='inout'])"><xsl:text>, </xsl:text></xsl:if>
		</xsl:for-each>
		<xsl:text>) {}; 
</xsl:text>
	</xsl:template>

	
	<xsl:template name="CrossCuttingAttribute">
		<xsl:param name="attribute"/>
		<xsl:param name="ontypes"/>
		
		<xsl:choose>
			<xsl:when test="contains($ontypes, ' ')">
				<xsl:variable name="head" select="substring-before($ontypes, ' ')"/>
				<xsl:variable name="body" select="substring-after($ontypes, ' ')"/>
				<xsl:call-template name="writeAttribute">
					<xsl:with-param name="attribute" select="$attribute"/>
					<xsl:with-param name="ontype" select="$head"/>
				</xsl:call-template>
				<xsl:call-template name="CrossCuttingAttribute">
					<xsl:with-param name="attribute" select="$attribute"/>
					<xsl:with-param name="ontypes" select="$body"/>
				</xsl:call-template>
			</xsl:when>
			<xsl:otherwise>
				<xsl:call-template name="writeAttribute">
					<xsl:with-param name="attribute" select="$attribute"/>
					<xsl:with-param name="ontype" select="$ontypes"/>
				</xsl:call-template>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>

	<xsl:template name="writeAttribute">
		<xsl:param name="attribute"/>
		<xsl:param name="ontype"/>

		<xsl:text>  </xsl:text>
		<xsl:if test="$attribute/@isStatic='true'">
			<xsl:text>static </xsl:text>
		</xsl:if>
		<xsl:value-of select="$attribute/@visibility"/><xsl:text> </xsl:text>
		<xsl:call-template name="writeType"><xsl:with-param name="type" select="$attribute/@type"/></xsl:call-template>
		<xsl:text> </xsl:text>

		<xsl:call-template name="buildName">
			<xsl:with-param name="element" select="//*[@xmi:id=$ontype]"/>
			<xsl:with-param name="name" select="$attribute/@name"/>			
		</xsl:call-template>
		<xsl:text>;
</xsl:text>
	</xsl:template>


	<xsl:template name="ProcessAdvice">
		<xsl:param name="operation"/>
		<xsl:param name="adviceexecution"/>
		<xsl:param name="pointcutID"/>

		<xsl:message terminate="no">Processing advice <xsl:value-of select="$operation/@name"/></xsl:message>		
		<xsl:text>  </xsl:text>
		<xsl:choose>
			<xsl:when test="$adviceexecution='AroundAdvice'">
				<xsl:if test="$operation/ownedParameter[@direction='return']">
					<xsl:choose>
						<xsl:when test="count($operation/ownedParameter[@direction='return']/type)=1">
							<xsl:value-of select="translate(normalize-space(translate(translate($operation/ownedParameter[@direction='return']/type/xmi:Extension/referenceExtension/@referentPath,' ','_'),':',' ')),' ','.')"/>
						</xsl:when>
						<xsl:when test="count($operation/ownedParameter[@direction='return']/@type)=1">
							<xsl:call-template name="writeType"><xsl:with-param name="type" select="$operation/ownedParameter[@direction='return']/@type"/></xsl:call-template>
						</xsl:when>
					</xsl:choose>
					<xsl:text> </xsl:text>
				</xsl:if>
				<xsl:text>around(</xsl:text>
				<xsl:for-each select="$operation/ownedParameter[@direction='in' or @direction='inout']">
					<xsl:choose>
						<xsl:when test="count(type)=1">
							<xsl:value-of select="translate(normalize-space(translate(translate(type/xmi:Extension/referenceExtension/@referentPath,' ','_'),':',' ')),' ','.')"/>
						</xsl:when>
						<xsl:when test="count(@type)=1">
							<xsl:call-template name="writeType"><xsl:with-param name="type" select="@type"/></xsl:call-template>
						</xsl:when>
					</xsl:choose>
					<xsl:text> </xsl:text><xsl:value-of select="@name"/>
					<xsl:if test="position() &lt; count(../ownedParameter[@direction='in' or @direction='inout'])"><xsl:text>, </xsl:text></xsl:if>
				</xsl:for-each>
				<xsl:text>) </xsl:text>
				<xsl:for-each select="raisedException">
					<xsl:text>throws </xsl:text>
					<xsl:variable name="exceptionID" select="@xmi:idref"/>
					<xsl:call-template name="buildName">
						<xsl:with-param name="element" select="//*[@xmi:id=$exceptionID]/.."/>
						<xsl:with-param name="name" select="//*[@xmi:id=$exceptionID]/@name"/>
					</xsl:call-template>
					<xsl:if test="position() &lt; count(../raisedException)"><xsl:text>, </xsl:text></xsl:if>
				</xsl:for-each>
			</xsl:when>
			<xsl:when test="$adviceexecution='BeforeAdvice'">
				<xsl:text>before(</xsl:text>
				<xsl:for-each select="$operation/ownedParameter[@direction='in' or @direction='inout']">
					<xsl:choose>
						<xsl:when test="count(type)=1">
							<xsl:value-of select="translate(normalize-space(translate(translate(type/xmi:Extension/referenceExtension/@referentPath,' ','_'),':',' ')),' ','.')"/>
						</xsl:when>
						<xsl:when test="count(@type)=1">
							<xsl:call-template name="writeType"><xsl:with-param name="type" select="@type"/></xsl:call-template>
						</xsl:when>
					</xsl:choose>
					<xsl:text> </xsl:text><xsl:value-of select="@name"/>
					<xsl:if test="position() &lt; count(../ownedParameter[@direction='in' or @direction='inout'])"><xsl:text>, </xsl:text></xsl:if>
				</xsl:for-each>
				<xsl:text>) </xsl:text>
				<xsl:for-each select="raisedException">
					<xsl:text>throws </xsl:text>
					<xsl:variable name="exceptionID" select="@xmi:idref"/>
					<xsl:call-template name="buildName">
						<xsl:with-param name="element" select="//*[@xmi:id=$exceptionID]/.."/>
						<xsl:with-param name="name" select="//*[@xmi:id=$exceptionID]/@name"/>
					</xsl:call-template>
					<xsl:if test="position() &lt; count(../raisedException)"><xsl:text>, </xsl:text></xsl:if>
				</xsl:for-each>			
			</xsl:when>
			<xsl:when test="$adviceexecution='AfterAdvice'">
				<xsl:text>after(</xsl:text>
				<xsl:for-each select="$operation/ownedParameter[@direction='in' or @direction='inout']">
					<xsl:choose>
						<xsl:when test="count(type)=1">
							<xsl:value-of select="translate(normalize-space(translate(translate(type/xmi:Extension/referenceExtension/@referentPath,' ','_'),':',' ')),' ','.')"/>
						</xsl:when>
						<xsl:when test="count(@type)=1">
							<xsl:call-template name="writeType"><xsl:with-param name="type" select="@type"/></xsl:call-template>
						</xsl:when>
					</xsl:choose>
					<xsl:text> </xsl:text><xsl:value-of select="@name"/>
					<xsl:if test="position() &lt; count(../ownedParameter[@direction='in' or @direction='inout'])"><xsl:text>, </xsl:text></xsl:if>
				</xsl:for-each>
				<xsl:text>) </xsl:text>
				<xsl:if test="$operation/ownedParameter[@direction='return']">
					<xsl:text>returning (</xsl:text> 
					<xsl:choose>
						<xsl:when test="count($operation/ownedParameter[@direction='return']/type)=1">
							<xsl:value-of select="translate(normalize-space(translate(translate($operation/ownedParameter[@direction='return']/type/xmi:Extension/referenceExtension/@referentPath,' ','_'),':',' ')),' ','.')"/>
						</xsl:when>
						<xsl:when test="count($operation/ownedParameter[@direction='return']/@type)=1">
							<xsl:call-template name="writeType"><xsl:with-param name="type" select="$operation/ownedParameter[@direction='return']/@type"/></xsl:call-template>
						</xsl:when>
					</xsl:choose>
					<xsl:text> </xsl:text>
					<xsl:choose>
						<xsl:when test="string-length($operation/ownedParameter[@direction='return']/@name) &gt; 0">
							<xsl:value-of select="$operation/ownedParameter[@direction='return']/@name"/>
						</xsl:when>
						<xsl:otherwise>
							<xsl:text>result</xsl:text>
						</xsl:otherwise>
					</xsl:choose>
					<xsl:text>) </xsl:text>
				</xsl:if>
				<xsl:if test="raisedException">
					<xsl:text>throwing (</xsl:text>
					<xsl:for-each select="raisedException">
						<xsl:variable name="exceptionID" select="@xmi:idref"/>
						<xsl:call-template name="buildName">
							<xsl:with-param name="element" select="//*[@xmi:id=$exceptionID]/.."/>
							<xsl:with-param name="name" select="//*[@xmi:id=$exceptionID]/@name"/>
						</xsl:call-template>
						<xsl:text> </xsl:text><xsl:value-of select="concat('e',position())"/>
						<xsl:if test="position() &lt; count(../raisedException)"><xsl:text>, </xsl:text></xsl:if>
					</xsl:for-each>
					<xsl:text>)</xsl:text>
				</xsl:if>
			</xsl:when>
		</xsl:choose>
		<xsl:text> : </xsl:text>
		<xsl:value-of select="//ownedAttribute[@xmi:id=$pointcutID]/@name"/>
		<xsl:text>(</xsl:text>
		<xsl:for-each select="$operation/ownedParameter[@direction='in' or @direction='inout']">
			<xsl:value-of select="@name"/>
			<xsl:if test="position() &lt; count(../ownedParameter[@direction='in' or @direction='inout'])"><xsl:text>, </xsl:text></xsl:if>
		</xsl:for-each>
		<xsl:text>) {}
</xsl:text>		
	</xsl:template>


	<xsl:template name="ProcessPointCut">
		<xsl:param name="attribute"/>
		<xsl:param name="pointcut"/>

		<xsl:variable name="pointcuttype" select="local-name($pointcut)"/>
		<xsl:message terminate="no">Processing <xsl:value-of select="$pointcuttype"/> pointcut <xsl:value-of select="$pointcut/@xmi:id"/> for attribute <xsl:value-of select="$attribute/@name"/></xsl:message>
		<xsl:text>  pointcut </xsl:text><xsl:value-of select="$attribute/@name"/>
		<xsl:text>(</xsl:text>
		<xsl:call-template name="WriteContext">
			<xsl:with-param name="pointcutlist" select="$attribute/@xmi:id"/>
			<xsl:with-param name="composition" select="''"/>
		</xsl:call-template>
		<xsl:text>): </xsl:text>
		<xsl:call-template name="WritePointCut">
			<xsl:with-param name="pointcutlist" select="$attribute/@xmi:id"/>
			<xsl:with-param name="composition" select="''"/>
		</xsl:call-template>		
		<xsl:text>;
</xsl:text>
	</xsl:template>
		
	<xsl:template name="WriteContext">
		<xsl:param name="pointcutlist"/>
		<xsl:param name="composition"/>
		<xsl:message terminate="no">Called writeContext with pointcutlist <xsl:value-of select="$pointcutlist"/> and composition <xsl:value-of select="$composition"/></xsl:message>
		<xsl:choose>
			<xsl:when test="contains($pointcutlist, ' ')">
				<xsl:variable name="head" select="substring-before($pointcutlist, ' ')"/>
				<xsl:variable name="body" select="substring-after($pointcutlist, ' ')"/>
				<xsl:variable name="pointcut" select="//aspectJ:*[@base_StructuralFeature=$head]"/>
				<xsl:variable name="attribute" select="//ownedAttribute[@xmi:id = $head]"/>
				<xsl:variable name="pointcuttype" select="local-name($pointcut)"/>
				
				<xsl:message terminate="no">Pointcut is <xsl:value-of select="$pointcut/@xmi:id"/> of type <xsl:value-of select="$pointcuttype"/></xsl:message>

				<xsl:if test="$pointcuttype='PointCutConjunction' or $pointcuttype='PointCutDisjunction'">
					<xsl:if test="$pointcuttype='PointCutConjunction'">
						<xsl:call-template name="WriteContext">
							<xsl:with-param name="pointcutlist" select="$pointcut/@composee"/>
							<xsl:with-param name="composition" select="and"/>
						</xsl:call-template>
					</xsl:if>
					<xsl:if test="$pointcuttype='PointCutDisjunction'">
						<xsl:call-template name="WriteContext">
							<xsl:with-param name="pointcutlist" select="$pointcut/@composee"/>
							<xsl:with-param name="composition" select="or"/>
						</xsl:call-template>
					</xsl:if>
				</xsl:if>

				<xsl:if test="$pointcuttype='ThisPointCut' or $pointcuttype='ArgsPointCut' or $pointcuttype='TargetPointCut'">
					<xsl:for-each select="$pointcut/type">
						<xsl:message terminate="no">Writing type ... </xsl:message>
						<xsl:variable name="pos" select="position()"/>
						<xsl:if test="@xmi:type='uml:PrimitiveType'">
							<xsl:value-of select="substring-after(@href, '#')"/>
						</xsl:if>
						<xsl:if test="@xmi:type='uml:Class' or @xmi:type='uml:Classifier' or @xmi:type='uml:Type'">
							<xsl:call-template name="writeType"><xsl:with-param name="type" select="substring-after(@href, '#')"/></xsl:call-template>
						</xsl:if>
						<xsl:text> </xsl:text>
						<xsl:value-of select="../argNames[position()=$pos]"/>
						<xsl:if test="$pos &lt; last()"><xsl:text>, </xsl:text></xsl:if>
					</xsl:for-each>
					<xsl:text>, </xsl:text>
				</xsl:if>
				
				<xsl:call-template name="WriteContext">
					<xsl:with-param name="pointcutlist" select="$body"/>
					<xsl:with-param name="composition" select="$composition"/>
				</xsl:call-template>
			</xsl:when>
			<xsl:otherwise>
				<xsl:variable name="pointcut" select="//aspectJ:*[@base_StructuralFeature=$pointcutlist]"/>
				<xsl:variable name="pointcuttype" select="local-name($pointcut)"/>
				<xsl:message terminate="no">Pointcut is <xsl:value-of select="$pointcut/@xmi:id"/>  of type <xsl:value-of select="$pointcuttype"/></xsl:message>

				<xsl:if test="$pointcuttype='PointCutConjunction' or $pointcuttype='PointCutDisjunction'">
					<xsl:if test="$pointcuttype='PointCutConjunction'">
						<xsl:call-template name="WriteContext">
							<xsl:with-param name="pointcutlist" select="$pointcut/@composee"/>
							<xsl:with-param name="composition" select="and"/>
						</xsl:call-template>
					</xsl:if>
					<xsl:if test="$pointcuttype='PointCutDisjunction'">
						<xsl:call-template name="WriteContext">
							<xsl:with-param name="pointcutlist" select="$pointcut/@composee"/>
							<xsl:with-param name="composition" select="or"/>
						</xsl:call-template>
					</xsl:if>
				</xsl:if>

				<xsl:if test="$pointcuttype='ThisPointCut' or $pointcuttype='ArgsPointCut' or $pointcuttype='TargetPointCut'">	
					<xsl:for-each select="$pointcut/type">
						<xsl:message terminate="no">Writing type ... </xsl:message>
						<xsl:variable name="pos" select="position()"/>
						<xsl:if test="@xmi:type='uml:PrimitiveType'">
							<xsl:value-of select="substring-after(@href, '#')"/>
						</xsl:if>
						<xsl:if test="@xmi:type='uml:Class' or @xmi:type='uml:Classifier' or @xmi:type='uml:Type'">
							<xsl:call-template name="writeType"><xsl:with-param name="type" select="substring-after(@href, '#')"/></xsl:call-template>
						</xsl:if>
						<xsl:text> </xsl:text>
						<xsl:value-of select="../argNames[position()=$pos]"/>
						<xsl:if test="$pos &lt; last()"><xsl:text>, </xsl:text></xsl:if>
					</xsl:for-each>
				</xsl:if>

			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>


	<xsl:template name="WritePointCut">
		<xsl:param name="pointcutlist"/>
		<xsl:param name="composition"/>
		
		<xsl:message terminate="no">Called writePointCut with pointcutlist <xsl:value-of select="$pointcutlist"/> and composition <xsl:value-of select="$composition"/></xsl:message>
		<xsl:choose>
			<xsl:when test="contains($pointcutlist, ' ')">
				<xsl:variable name="head" select="substring-before($pointcutlist, ' ')"/>
				<xsl:variable name="body" select="substring-after($pointcutlist, ' ')"/>
				<xsl:variable name="pointcut" select="//aspectJ:*[@base_StructuralFeature=$head]"/>
				<xsl:variable name="attribute" select="//ownedAttribute[@xmi:id = $head]"/>
				<xsl:variable name="pointcuttype" select="local-name($pointcut)"/>
				
				<xsl:message terminate="no">Pointcut is <xsl:value-of select="$pointcut/@xmi:id"/> of type <xsl:value-of select="$pointcuttype"/></xsl:message>

				<xsl:if test="$pointcuttype='PointCutNegation'">
					<xsl:text>!</xsl:text>
					<xsl:call-template name="WritePointCut">
						<xsl:with-param name="pointcutlist" select="$pointcut/@negates"/>
						<xsl:with-param name="composition" select="''"/>
					</xsl:call-template>
				</xsl:if>
				
				<xsl:if test="$pointcuttype='PointCutConjunction' or $pointcuttype='PointCutDisjunction'">
					<xsl:if test="$pointcuttype='PointCutConjunction'">
						<xsl:text>(</xsl:text>
						<xsl:call-template name="WritePointCut">
							<xsl:with-param name="pointcutlist" select="$pointcut/@composee"/>
							<xsl:with-param name="composition" select="'and'"/>
						</xsl:call-template>
						<xsl:text>)</xsl:text>
					</xsl:if>
					<xsl:if test="$pointcuttype='PointCutDisjunction'">
						<xsl:text>(</xsl:text>
						<xsl:call-template name="WritePointCut">
							<xsl:with-param name="pointcutlist" select="$pointcut/@composee"/>
							<xsl:with-param name="composition" select="'or'"/>
						</xsl:call-template>
						<xsl:text>)</xsl:text>
					</xsl:if>
				</xsl:if>

				<xsl:if test="$pointcuttype='ThisPointCut'">
					<xsl:message terminate="no">Writing ThisPointCut ... </xsl:message>
					<xsl:text>this(</xsl:text>
					<xsl:for-each select="$pointcut/type">
						<xsl:variable name="pos" select="position()"/>
						<xsl:if test="@xmi:type='uml:PrimitiveType'">
							<xsl:value-of select="substring-after(@href, '#')"/>
						</xsl:if>
						<xsl:if test="@xmi:type='uml:Class' or @xmi:type='uml:Classifier' or @xmi:type='uml:Type'">
							<xsl:call-template name="writeType"><xsl:with-param name="type" select="substring-after(@href, '#')"/></xsl:call-template>
						</xsl:if>
						<xsl:text> </xsl:text>
						<xsl:value-of select="../argNames[position()=$pos]"/>
						<xsl:if test="$pos &lt; last()"><xsl:text>, </xsl:text></xsl:if>
					</xsl:for-each>
					<xsl:text>)</xsl:text>
				</xsl:if>

				<xsl:if test="$pointcuttype='ArgsPointCut'">
					<xsl:message terminate="no">Writing ArgsPointCut ... </xsl:message>
					<xsl:text>args(</xsl:text>
					<xsl:for-each select="$pointcut/type">
						<xsl:variable name="pos" select="position()"/>
						<xsl:if test="@xmi:type='uml:PrimitiveType'">
							<xsl:value-of select="substring-after(@href, '#')"/>
						</xsl:if>
						<xsl:if test="@xmi:type='uml:Class' or @xmi:type='uml:Classifier' or @xmi:type='uml:Type'">
							<xsl:call-template name="writeType"><xsl:with-param name="type" select="substring-after(@href, '#')"/></xsl:call-template>
						</xsl:if>
						<xsl:text> </xsl:text>
						<xsl:value-of select="../argNames[position()=$pos]"/>
						<xsl:if test="$pos &lt; last()"><xsl:text>, </xsl:text></xsl:if>
					</xsl:for-each>
					<xsl:text>)</xsl:text>
				</xsl:if>
				
				<xsl:if test="$pointcuttype='TargetPointCut'">
					<xsl:message terminate="no">Writing TargetPointCut ... </xsl:message>
					<xsl:text>target(</xsl:text>
					<xsl:for-each select="$pointcut/type">
						<xsl:variable name="pos" select="position()"/>
						<xsl:if test="@xmi:type='uml:PrimitiveType'">
							<xsl:value-of select="substring-after(@href, '#')"/>
						</xsl:if>
						<xsl:if test="@xmi:type='uml:Class' or @xmi:type='uml:Classifier' or @xmi:type='uml:Type'">
							<xsl:call-template name="writeType"><xsl:with-param name="type" select="substring-after(@href, '#')"/></xsl:call-template>
						</xsl:if>
						<xsl:text> </xsl:text>
						<xsl:value-of select="../argNames[position()=$pos]"/>
						<xsl:if test="$pos &lt; last()"><xsl:text>, </xsl:text></xsl:if>
					</xsl:for-each>
					<xsl:text>)</xsl:text>
				</xsl:if>

				<xsl:choose>
					<xsl:when test="$pointcuttype='WithinCodePointCut'">
						<xsl:call-template name="writeOperations">
							<xsl:with-param name="operations" select="$pointcut/@operation"/>
							<xsl:with-param name="pointcuttype" select="'withincode'"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:when test="$pointcuttype='ExecutionPointCut'">
						<xsl:call-template name="writeOperations">
							<xsl:with-param name="operations" select="$pointcut/@operation"/>
							<xsl:with-param name="pointcuttype" select="'execution'"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:when test="$pointcuttype='PreInitializationPointCut'">
						<xsl:call-template name="writeOperations">
							<xsl:with-param name="operations" select="$pointcut/@operation"/>
							<xsl:with-param name="pointcuttype" select="'preinitialization'"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:when test="$pointcuttype='InitializationPointCut'">
						<xsl:call-template name="writeOperations">
							<xsl:with-param name="operations" select="$pointcut/@operation"/>
							<xsl:with-param name="pointcuttype" select="'initialization'"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:when test="$pointcuttype='CallPointCut'">
						<xsl:call-template name="writeOperations">
							<xsl:with-param name="operations" select="$pointcut/@operation"/>
							<xsl:with-param name="pointcuttype" select="'call'"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:when test="$pointcuttype='SetPointCut'">
						<xsl:call-template name="writeFields">
							<xsl:with-param name="fields" select="$pointcut/@field"/>
							<xsl:with-param name="pointcuttype" select="'set'"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:when test="$pointcuttype='GetPointCut'">
						<xsl:call-template name="writeFields">
							<xsl:with-param name="fields" select="$pointcut/@operation"/>
							<xsl:with-param name="pointcuttype" select="'get'"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:when test="$pointcuttype='StaticInitializationPointCut'">
						<xsl:call-template name="writeTypes">
							<xsl:with-param name="types" select="$pointcut/@type"/>
							<xsl:with-param name="pointcuttype" select="'staticinitialization'"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:when test="$pointcuttype='ExceptionPointCut'">
						<xsl:call-template name="writeTypes">
							<xsl:with-param name="types" select="$pointcut/@type"/>
							<xsl:with-param name="pointcuttype" select="'handler'"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:when test="$pointcuttype='WithinPointCut'">
						<xsl:call-template name="writeTypes">
							<xsl:with-param name="types" select="$pointcut/@type"/>
							<xsl:with-param name="pointcuttype" select="'within'"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:when test="$pointcuttype='AdviceExecutionPointCut'">
						<xsl:text>adviceexecution()</xsl:text>
					</xsl:when>
					<xsl:when test="$pointcuttype='CFlowPointCut'">
						<xsl:text>cflow(</xsl:text>
						<xsl:value-of select="//ownedAttribute[@xmi:id=$pointcut/@selectedPointCut]/@name"/>
						<xsl:text>)</xsl:text>
					</xsl:when>
					<xsl:when test="$pointcuttype='CFlowBelowPointCut'">
						<xsl:text>cflowbelow(</xsl:text>
						<xsl:value-of select="//ownedAttribute[@xmi:id=$pointcut/@selectedPointCut]/@name"/>
						<xsl:text>)</xsl:text>
					</xsl:when>
				</xsl:choose>
				
				<xsl:if test="$composition='and'">
					<xsl:text> &amp;&amp; </xsl:text>
				</xsl:if>
				<xsl:if test="$composition='or'">
					<xsl:text> || </xsl:text>
				</xsl:if>

				<xsl:call-template name="WritePointCut">
					<xsl:with-param name="pointcutlist" select="$body"/>
					<xsl:with-param name="composition" select="$composition"/>
				</xsl:call-template>

			</xsl:when>
			<xsl:otherwise>
				<xsl:variable name="pointcut" select="//aspectJ:*[@base_StructuralFeature=$pointcutlist]"/>
				<xsl:variable name="pointcuttype" select="local-name($pointcut)"/>
				<xsl:message terminate="no">Pointcut is <xsl:value-of select="$pointcut/@xmi:id"/> of type <xsl:value-of select="$pointcuttype"/></xsl:message>

				<xsl:if test="$pointcuttype='PointCutNegation'">
					<xsl:text>!</xsl:text>
					<xsl:call-template name="WritePointCut">
						<xsl:with-param name="pointcutlist" select="$pointcut/@negates"/>
						<xsl:with-param name="composition" select="''"/>
					</xsl:call-template>
				</xsl:if>
				
				<xsl:if test="$pointcuttype='PointCutConjunction' or $pointcuttype='PointCutDisjunction'">
					<xsl:if test="$pointcuttype='PointCutConjunction'">
						<xsl:text>(</xsl:text>
						<xsl:call-template name="WritePointCut">
							<xsl:with-param name="pointcutlist" select="$pointcut/@composee"/>
							<xsl:with-param name="composition" select="'and'"/>
						</xsl:call-template>
						<xsl:text>)</xsl:text>
					</xsl:if>
					<xsl:if test="$pointcuttype='PointCutDisjunction'">
						<xsl:text>(</xsl:text>
						<xsl:call-template name="WritePointCut">
							<xsl:with-param name="pointcutlist" select="$pointcut/@composee"/>
							<xsl:with-param name="composition" select="'or'"/>
						</xsl:call-template>
						<xsl:text>)</xsl:text>
					</xsl:if>
				</xsl:if>

				<xsl:if test="$pointcuttype='ThisPointCut'">
					<xsl:message terminate="no">Writing ThisPointCut ... </xsl:message>
					<xsl:text>this(</xsl:text>
					<xsl:for-each select="$pointcut/type">
						<xsl:variable name="pos" select="position()"/>
						<xsl:if test="@xmi:type='uml:PrimitiveType'">
							<xsl:value-of select="substring-after(@href, '#')"/>
						</xsl:if>
						<xsl:if test="@xmi:type='uml:Class' or @xmi:type='uml:Classifier' or @xmi:type='uml:Type'">
							<xsl:call-template name="writeType"><xsl:with-param name="type" select="substring-after(@href, '#')"/></xsl:call-template>
						</xsl:if>
						<xsl:text> </xsl:text>
						<xsl:value-of select="../argNames[position()=$pos]"/>
						<xsl:if test="$pos &lt; last()"><xsl:text>, </xsl:text></xsl:if>
					</xsl:for-each>
					<xsl:text>)</xsl:text>
				</xsl:if>

				<xsl:if test="$pointcuttype='ArgsPointCut'">
					<xsl:message terminate="no">Writing ArgsPointCut ... </xsl:message>
					<xsl:text>args(</xsl:text>
					<xsl:for-each select="$pointcut/type">
						<xsl:variable name="pos" select="position()"/>
						<xsl:if test="@xmi:type='uml:PrimitiveType'">
							<xsl:value-of select="substring-after(@href, '#')"/>
						</xsl:if>
						<xsl:if test="@xmi:type='uml:Class' or @xmi:type='uml:Classifier' or @xmi:type='uml:Type'">
							<xsl:call-template name="writeType"><xsl:with-param name="type" select="substring-after(@href, '#')"/></xsl:call-template>
						</xsl:if>
						<xsl:text> </xsl:text>
						<xsl:value-of select="../argNames[position()=$pos]"/>
						<xsl:if test="$pos &lt; last()"><xsl:text>, </xsl:text></xsl:if>
					</xsl:for-each>
					<xsl:text>)</xsl:text>
				</xsl:if>
				
				<xsl:if test="$pointcuttype='TargetPointCut'">
					<xsl:message terminate="no">Writing TargetPointCut ... </xsl:message>
					<xsl:text>target(</xsl:text>
					<xsl:for-each select="$pointcut/type">
						<xsl:variable name="pos" select="position()"/>
						<xsl:if test="@xmi:type='uml:PrimitiveType'">
							<xsl:value-of select="substring-after(@href, '#')"/>
						</xsl:if>
						<xsl:if test="@xmi:type='uml:Class' or @xmi:type='uml:Classifier' or @xmi:type='uml:Type'">
							<xsl:call-template name="writeType"><xsl:with-param name="type" select="substring-after(@href, '#')"/></xsl:call-template>
						</xsl:if>
						<xsl:text> </xsl:text>
						<xsl:value-of select="../argNames[position()=$pos]"/>
						<xsl:if test="$pos &lt; last()"><xsl:text>, </xsl:text></xsl:if>
					</xsl:for-each>
					<xsl:text>)</xsl:text>
				</xsl:if>

				<xsl:choose>
					<xsl:when test="$pointcuttype='WithinCodePointCut'">
						<xsl:call-template name="writeOperations">
							<xsl:with-param name="operations" select="$pointcut/@operation"/>
							<xsl:with-param name="pointcuttype" select="'withincode'"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:when test="$pointcuttype='ExecutionPointCut'">
						<xsl:call-template name="writeOperations">
							<xsl:with-param name="operations" select="$pointcut/@operation"/>
							<xsl:with-param name="pointcuttype" select="'execution'"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:when test="$pointcuttype='PreInitializationPointCut'">
						<xsl:call-template name="writeOperations">
							<xsl:with-param name="operations" select="$pointcut/@operation"/>
							<xsl:with-param name="pointcuttype" select="'preinitialization'"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:when test="$pointcuttype='InitializationPointCut'">
						<xsl:call-template name="writeOperations">
							<xsl:with-param name="operations" select="$pointcut/@operation"/>
							<xsl:with-param name="pointcuttype" select="'initialization'"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:when test="$pointcuttype='CallPointCut'">
						<xsl:call-template name="writeOperations">
							<xsl:with-param name="operations" select="$pointcut/@operation"/>
							<xsl:with-param name="pointcuttype" select="'call'"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:when test="$pointcuttype='SetPointCut'">
						<xsl:call-template name="writeFields">
							<xsl:with-param name="fields" select="$pointcut/@field"/>
							<xsl:with-param name="pointcuttype" select="'set'"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:when test="$pointcuttype='GetPointCut'">
						<xsl:call-template name="writeFields">
							<xsl:with-param name="fields" select="$pointcut/@operation"/>
							<xsl:with-param name="pointcuttype" select="'get'"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:when test="$pointcuttype='StaticInitializationPointCut'">
						<xsl:call-template name="writeTypes">
							<xsl:with-param name="types" select="$pointcut/@type"/>
							<xsl:with-param name="pointcuttype" select="'staticinitialization'"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:when test="$pointcuttype='ExceptionPointCut'">
						<xsl:call-template name="writeTypes">
							<xsl:with-param name="types" select="$pointcut/@type"/>
							<xsl:with-param name="pointcuttype" select="'handler'"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:when test="$pointcuttype='WithinPointCut'">
						<xsl:call-template name="writeTypes">
							<xsl:with-param name="types" select="$pointcut/@type"/>
							<xsl:with-param name="pointcuttype" select="'within'"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:when test="$pointcuttype='AdviceExecutionPointCut'">
						<xsl:text>adviceexecution()</xsl:text>
					</xsl:when>
					<xsl:when test="$pointcuttype='CFlowPointCut'">
						<xsl:text>cflow(</xsl:text>
						<xsl:value-of select="//ownedAttribute[@xmi:id=$pointcut/@selectedPointCut]/@name"/>
						<xsl:text>)</xsl:text>
					</xsl:when>
					<xsl:when test="$pointcuttype='CFlowBelowPointCut'">
						<xsl:text>cflowbelow(</xsl:text>
						<xsl:value-of select="//ownedAttribute[@xmi:id=$pointcut/@selectedPointCut]/@name"/>
						<xsl:text>)</xsl:text>
					</xsl:when>
				</xsl:choose>

			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>

	<xsl:template name="writeOperations">
		<xsl:param name="operations"/>
		<xsl:param name="pointcuttype"/>
		
		<xsl:message terminate="no">WriteOperations called with <xsl:value-of select="$operations"/></xsl:message>

		<xsl:choose>
			<xsl:when test="contains($operations, ' ')">
				<xsl:variable name="head" select="substring-before($operations, ' ')"/>
				<xsl:variable name="tail" select="substring-after($operations, ' ')"/>
				<xsl:value-of select="$pointcuttype"/>
				<xsl:text>(</xsl:text>
				<xsl:call-template name="writePointCutOperation">
					<xsl:with-param name="operation" select="//*[@xmi:id=$head]"/>
				</xsl:call-template>
				<xsl:text>) || </xsl:text>
				<xsl:call-template name="writeOperations">
					<xsl:with-param name="operations" select="$tail"/>
					<xsl:with-param name="pointcuttype" select="$pointcuttype"/>
				</xsl:call-template>			
			</xsl:when>
			<xsl:otherwise>
				<xsl:value-of select="$pointcuttype"/>
				<xsl:text>(</xsl:text>
				<xsl:call-template name="writePointCutOperation">
					<xsl:with-param name="operation" select="//*[@xmi:id=$operations]"/>
				</xsl:call-template>
				<xsl:text>)</xsl:text>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
	<xsl:template name="writePointCutOperation">
		<xsl:param name="operation"/>
		
		<xsl:variable name="ontype" select="$operation/../@xmi:id"/>
		<xsl:message terminate="no">Writing pointcut operation <xsl:value-of select="$operation/@name"/> on type <xsl:value-of select="$ontype"/></xsl:message>
		
		<xsl:value-of select="$operation/@visibility"/><xsl:text> </xsl:text>
		<xsl:choose>
			<xsl:when test="count($operation/ownedParameter[@direction='return'])=1">
				<xsl:choose>
					<xsl:when test="count($operation/ownedParameter[@direction='return']/type)=1">
						<xsl:value-of select="translate(normalize-space(translate(translate($operation/ownedParameter[@direction='return']/type/xmi:Extension/referenceExtension/@referentPath,' ','_'),':',' ')),' ','.')"/>
					</xsl:when>
					<xsl:when test="count($operation/ownedParameter[@direction='return']/@type)=1">
						<xsl:call-template name="writeType"><xsl:with-param name="type" select="$operation/ownedParameter[@direction='return']/@type"/></xsl:call-template>
					</xsl:when>
				</xsl:choose>
				<xsl:text> </xsl:text>
			</xsl:when>
			<xsl:otherwise>
				<xsl:text>void </xsl:text>
			</xsl:otherwise>
		</xsl:choose>
		<xsl:call-template name="buildName">
			<xsl:with-param name="element" select="//*[@xmi:id=$ontype]"/>
			<xsl:with-param name="name" select="$operation/@name"/>			
		</xsl:call-template>
		<xsl:text> (</xsl:text>
		<xsl:for-each select="$operation/ownedParameter[@direction='in' or @direction='inout']">
			<xsl:choose>
				<xsl:when test="count(@type)=1">
					<xsl:call-template name="writeType"><xsl:with-param name="type" select="@type"/></xsl:call-template>
				</xsl:when>
				<xsl:when test="count(type)=1">
					<xsl:value-of select="translate(normalize-space(translate(translate(type/xmi:Extension/referenceExtension/@referentPath,' ','_'),':',' ')),' ','.')"/>
				</xsl:when>
			</xsl:choose>
			<xsl:text> </xsl:text><xsl:value-of select="@name"/>
			<xsl:if test="position() &lt; count(../ownedParameter[@direction='in' or @direction='inout'])"><xsl:text>, </xsl:text></xsl:if>
		</xsl:for-each>
		<xsl:text>)</xsl:text>
	</xsl:template>
	
	<xsl:template name="writeFields">
		<xsl:param name="fields"/>
		<xsl:param name="pointcuttype"/>
		
		<xsl:message terminate="no">WriteFields called with <xsl:value-of select="$fields"/></xsl:message>

		<xsl:choose>
			<xsl:when test="contains($fields, ' ')">
				<xsl:variable name="head" select="substring-before($fields, ' ')"/>
				<xsl:variable name="tail" select="substring-after($fields, ' ')"/>
				<xsl:value-of select="$pointcuttype"/>
				<xsl:text>(</xsl:text>
				<xsl:call-template name="writePointCutField">
					<xsl:with-param name="field" select="//*[@xmi:id=$head]"/>
				</xsl:call-template>
				<xsl:text>) || </xsl:text>
				<xsl:call-template name="writeFields">
					<xsl:with-param name="fields" select="$tail"/>
					<xsl:with-param name="pointcuttype" select="$pointcuttype"/>
				</xsl:call-template>			
			</xsl:when>
			<xsl:otherwise>
				<xsl:value-of select="$pointcuttype"/>
				<xsl:text>(</xsl:text>
				<xsl:call-template name="writePointCutField">
					<xsl:with-param name="field" select="//*[@xmi:id=$fields]"/>
				</xsl:call-template>
				<xsl:text>)</xsl:text>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
	<xsl:template name="writePointCutField">
		<xsl:param name="field"/>

		<xsl:if test="$field/@isStatic='true'">
			<xsl:text>static </xsl:text>
		</xsl:if>
		<xsl:value-of select="$field/@visibility"/><xsl:text> </xsl:text>

		<xsl:call-template name="buildName">
			<xsl:with-param name="element" select="$field/.."/>
			<xsl:with-param name="name" select="$field/@name"/>			
		</xsl:call-template>
	</xsl:template>
	
	
	<xsl:template name="writeTypes">
		<xsl:param name="types"/>
		<xsl:param name="pointcuttype"/>
		<xsl:message terminate="no">WriteTypes called with <xsl:value-of select="$types"/></xsl:message>

		<xsl:choose>
			<xsl:when test="contains($types, ' ')">
				<xsl:variable name="head" select="substring-before($types, ' ')"/>
				<xsl:variable name="tail" select="substring-after($types, ' ')"/>
				<xsl:value-of select="$pointcuttype"/>
				<xsl:text>(</xsl:text>
				<xsl:call-template name="writeType">
					<xsl:with-param name="type" select="$head"/>
				</xsl:call-template>
				<xsl:text>) || </xsl:text>
				<xsl:call-template name="writeTypes">
					<xsl:with-param name="types" select="$tail"/>
					<xsl:with-param name="pointcuttype" select="$pointcuttype"/>
				</xsl:call-template>			
			</xsl:when>
			<xsl:otherwise>
				<xsl:value-of select="$pointcuttype"/>
				<xsl:text>(</xsl:text>
				<xsl:call-template name="writeType">
					<xsl:with-param name="type" select="$types"/>
				</xsl:call-template>
				<xsl:text>)</xsl:text>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>

</xsl:stylesheet>

